#!/usr/bin/env bash
set -e
set -x

usage()
{
cat << EOF
usage: $0 options

OPTIONS:
   -h          Show this message
   -f          File in current directory
   -r          Repository on dist.yandex.ru
   -p          Version postfix
   -a          "0" disable autorestart service. Default "1"
   -e | -E     Script to download and run after package extraction
EOF
}

#url=''
repository=''
version_postfix='yandex0'
autorestart='1'

while getopts 'f:r:p:a:e:E:' OPTION
do
     case $OPTION in
         h)
             usage
             exit 1
             ;;
         f)
             file=$OPTARG
             ;;
         r)
             repository=$OPTARG
             ;;
         p)
             version_postfix=$OPTARG
             ;;
         a)
             autorestart=$OPTARG
             ;;
         e)
             postextract_hook=$OPTARG
             ;;
         E)
             postextract_hook_file=$OPTARG
             ;;
         ?)
             usage
             exit
             ;;
     esac
done

# Check that only -f or -u given
#if [[ -z $url ]] && [[ -z $file ]]; then
#     usage
#     exit 1
#fi
#if [[ ! -z $url ]] && [[ ! -z $file ]]; then
#     usage
#     exit 1
#fi
if [[ -z $file ]]; then
     usage
     exit 1
fi

if [[ -z $repository ]]; then
     usage
     exit 1
fi

#if [[ ! -z $url ]]; then
#     spackage=$(echo "$url" | awk -F '/' '{print $NF}')
#fi
if [[ ! -z $file ]]; then
     spackage=${file}
fi

package=$(echo $spackage | awk -F '_' '{print $1}')
version=$(echo $spackage | awk -F '_' '{print $2}')

#if [[ ! -z $url ]]; then
#     echo "Download package"
#     wget "$url"
#fispackage

echo "Extract package content"
dpkg-deb -x $spackage ${package}/

echo "Extract package control"
dpkg-deb -e $spackage ${package}/DEBIAN

if [ ! -z $postextract_hook ]; then
    echo "Run postextract hook"
    wget -O "postextract_hook.sh" "$postextract_hook"
    bash -x postextract_hook.sh
fi

if [ ! -z $postextract_hook_file ]; then
    echo "Run postextract hook"
    bash -x "$postextract_hook_file"
fi

echo "Extract package changelog"
changelog="${package}/usr/share/doc/${package}/changelog.Debian.gz"
if [[ ! -f $changelog ]]; then
   changelog=$(find ${package}/usr/share/doc -name 'changelog*.gz' 2>/dev/null| head -n 1)
else
    echo "changelog not found"
fi
if [[ -z $changelog ]] || [[ ! -f $changelog ]]; then
    rm -f $changelog
    dch --create --force-distribution --distribution "unstable" --package "$package" --newversion "${version}+${version_postfix}" -c changelog 'Rebuild package'
    gzip -c changelog > changelog.gz
    changelog="${package}/usr/share/doc/${package}/changelog.gz"
    mkdir -p $(dirname ${changelog})
    mv changelog.gz $changelog
else
    zcat $changelog > changelog
    echo "Update changelog"
    dch -b --package "$package" --newversion "${version}+${version_postfix}" -c changelog 'Rebuild package' --distribution "unstable"
    gzip -c changelog > $changelog
fi

# replace md5sum for changelog in md5sums file
md5file="${package}/DEBIAN/md5sums"
if [[ -f $md5file ]] && [[ -f $changelog ]]; then
    changelog_base=$(basename $changelog)
    if grep -q $changelog_base $md5file 2>/dev/null; then
	md5_new=$(md5sum $changelog | cut -f1 -d' ' )
        md5_old=$(grep $changelog_base $md5file  | cut -f1 -d' ')
        sed -i $md5file -e "s/$md5_old/$md5_new/g"
    else
	md5sum $changelog >> $md5file
	sed -i $md5file -e "s#${package}/##g"
    fi
else
    echo "md5file and changelog not found"
fi

if [ $autorestart -eq 0 ]; then
    echo "Disable restart"
    for f in $(ls ${package}/DEBIAN/{pre,post}*); do
        sed -i 's/invoke-rc.d/: #/' $f
    done
fi

# fix control
section=$(awk '/Section:/ {print $NF}' ${package}/DEBIAN/control 2>/dev/null || echo 'misc')
priority=$(awk '/Priority:/ {print $NF}' ${package}/DEBIAN/control 2>/dev/null || echo 'extra')
echo "$(echo $spackage | sed "s/$version/$version+$version_postfix/") $section $priority" > files
echo "Source: ${package}" > control
grep -iv "Source:" ${package}/DEBIAN/control >> control
cp control ${package}/DEBIAN/control

echo "Build package"
sed -i "s/$version/$version+$version_postfix/g" ${package}/DEBIAN/control

# Make root:root files' owner when repacking as user
if [[ $(whoami) != "root" ]]; then
     sudo chown -R root:root ${package}/
fi


dpkg-deb -b ${package}/ ./

echo "Generate changes"
dpkg-genchanges -b -u'.' -f'files' -c'control' -l'changelog' > $(echo $spackage | sed "s/$version/$version+$version_postfix/;s/\.deb/\.changes/")

echo "Sign package"
gpg --list-keys
debsign -k$EMAIL *.changes

echo "Dupload package"
dupload --to $repository -f --nomail *.changes

echo "Remove artifact"
sudo rm -rf ${package} ${spackage} control changelog files *.changes 
