# -*- coding: utf-8 -*-
import logging
import shlex

import os
import shutil

from sandbox import sdk2
import sandbox.common.types.resource as ctr
from sandbox.projects.sandbox.sandbox_lxc_image import RichTextTaskFailure
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.market.resources import MARKET_PBUILDER_IMAGE
from sandbox.projects.market.sre.MarketDebuilder import SUPPORTED_UBUNTU_VERSIONS, DEFAULT_UBUNTU_VERSION


class BuildMarketPbuilderImage(sdk2.Task):
    """ Task for building pbuild image """

    OWN_LOGGER_NAME = "shell"
    OWN_LOG_FILE_PATH = "shell.log"

    class Requirements(sdk2.Task.Requirements):
        privileged = True

    class Parameters(sdk2.Task.Parameters):
        distr_name = sdk2.parameters.String(
            'distr name',
            choices=zip(SUPPORTED_UBUNTU_VERSIONS, SUPPORTED_UBUNTU_VERSIONS),
            default=DEFAULT_UBUNTU_VERSION,
            required=True
        )
        last_image = sdk2.parameters.LastReleasedResource(
            'pbuilder image for update',
            resource_type=MARKET_PBUILDER_IMAGE,
            state=ctr.State.READY,
            attrs=dict(release='trusty'),
        )

    @property
    def exec_logger(self):
        try:
            return self.__exec_logger
        except AttributeError:
            exec_logger = logging.getLogger(self.OWN_LOGGER_NAME)
            map(exec_logger.removeHandler, exec_logger.handlers[:])
            handler = logging.FileHandler(str(self.log_path(self.OWN_LOG_FILE_PATH)))
            handler.setFormatter(logging.Formatter("%(asctime)s\t%(message)s"))
            exec_logger.addHandler(handler)
            exec_logger.propagate = False
            # noinspection PyAttributeOutsideInit
            self.__exec_logger = exec_logger

        return self.__exec_logger

    def _execute(self, command_line):
        args = shlex.split(command_line)
        my_env = os.environ.copy()
        my_env["TMPDIR"] = "/var/tmp"

        self.exec_logger.info("\t>>> EXECUTING COMMAND: %s", command_line)
        with sdk2.helpers.ProcessLog(self, logger=self.exec_logger, set_action=False) as pl:
            try:
                returncode = sp.Popen(args, stdout=pl.stdout, stderr=sp.STDOUT, env=my_env).wait()
                if returncode == 0:
                    return True
                raise RichTextTaskFailure(
                    "Command {!r} failed, see the log below.".format(command_line),
                    "Shell commands output written to <b><a href='{}'>{}</a></b>".format(
                        "/".join((self.log_resource.http_proxy, self.OWN_LOG_FILE_PATH)),
                        self.OWN_LOG_FILE_PATH
                    )
                )
            except Exception:
                logging.exception("SUBPROCESS ERROR")
                raise

    def on_execute(self):
        self._execute('apt-get update -qq')
        self._execute("apt-get install pbuilder -q0 --assume-yes --force-yes -o Dpkg::Options::='--force-confdef'")
        image_resource_path = sdk2.ResourceData(self.Parameters.last_image).path
        new_image_path = os.path.join(str(self.path()), image_resource_path.name)
        shutil.copy(str(image_resource_path), new_image_path)
        self._execute('pbuilder --update --basetgz {}'.format(new_image_path))
        sdk2.ResourceData(MARKET_PBUILDER_IMAGE(
             self, 'image for pbuilder', image_resource_path.name, release=self.Parameters.distr_name
        )).ready()
