# -*- coding: utf-8 -*-

import logging
import os
import tarfile

from sandbox.projects import resource_types
from sandbox.projects.common import constants
from sandbox.projects.common import utils
from sandbox.projects.common.build.parameters import ArcadiaUrl
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import sandboxapi
from sandbox.sandboxsdk import svn
from sandbox.sandboxsdk.task import SandboxTask

TARGET_RESOURCES = [
    # (source, resource_path, resource_type)
    ('tmpl', 'config.tar.gz', resource_types.MARKET_SERVICE_CONFIG),
    ('instancectl.conf', 'instancectl.conf', resource_types.MARKET_SERVICE_INSTANCECTL_CONFIG),
]


def create_link_on_service(service_id):
    return '<a href={0}>{0}</a>'.format(
        'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(service_id))


class ServicePath(parameters.SandboxStringParameter):
    name = 'service_path'
    description = 'Service path'
    required = True


class ReleaseService(parameters.SandboxBoolParameter):
    name = 'release_service'
    description = 'Release service'
    default_value = False
    sub_fields = {
        'true': ['service_ids'],
    }


class ServiceIDS(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'service_ids'
    description = 'Service IDs with environment'
    choices = [(x, x) for x in sandboxapi.RELEASE_STATUSES if x != sandboxapi.RELEASE_CANCELLED]


class BuildMarketServiceConfig(SandboxTask):
    """
    This task prepares configuration files for Market's services.
    The service configuration consist of the following files:
    - static file instancectl.conf
    - jinja2 templates archived into config.tar.gz
    """

    type = 'BUILD_MARKET_SERVICE_CONFIG'

    input_parameters = [
        ArcadiaUrl,
        ServicePath,
        ReleaseService,
        ServiceIDS,
    ]

    def on_execute(self):
        service_path = self.ctx.get(ServicePath.name, "")
        if not service_path or not service_path.startswith('/market/sre'):
            raise errors.SandboxTaskFailureError(
                "Service path '{}' should start with '/market/sre'".format(service_path))

        svn_url = svn.Arcadia.append(self.ctx[constants.ARCADIA_URL_KEY], service_path)
        path_to = '.'
        logging.info("Export '{}' to '{}'".format(svn_url, path_to))
        svn.Arcadia.export(svn_url, path_to)

        logging.info('Check sources of target resources exist')
        for source, _, _ in TARGET_RESOURCES:
            if not os.path.exists(source):
                raise errors.SandboxTaskFailureError(
                    "Path '{}' does not exist".format(service_path)
                )

        for source, path in [(s, p) for s, p, t in TARGET_RESOURCES if t == resource_types.MARKET_SERVICE_CONFIG]:
            logging.info("Create archive '{}'".format(path))
            with tarfile.open(name=path, mode='w:gz') as tar:
                tar.add(source)

        for _, path, _type in TARGET_RESOURCES:
            logging.info("Create resource '{}'".format(_type))
            resource = self.create_resource(
                description="file '{}' type '{}'".format(path, _type),
                resource_type=_type,
                resource_path=path,
            )
            self.mark_resource_ready(resource.id)

    def _release_service(self, service_id):
        logging.info('Release service {}'.format(service_id))
        token = self.get_vault_data('MARKETDEPLOY', 'nanny_token')
        nanny_client = nanny.NannyClient('http://nanny.yandex-team.ru/', token)

        try:
            nanny_client.get_service_current_state(service_id)
        except Exception as err:
            raise errors.SandboxTaskFailureError(
                'Failed to get state of service {}\n{}'.format(service_id, err))

        nanny_client.update_service_sandbox_resources(service_id, self.type, str(self.id))
        self.set_info(create_link_on_service(service_id), do_escape=False)

    def on_release(self, additional_parameters):
        SandboxTask.on_release(self, additional_parameters)

        release_status = additional_parameters.get('release_status')
        service_ids = utils.get_or_default(self.ctx, ServiceIDS)

        for service_id in [k for k, v in service_ids.iteritems() if v == release_status]:
            self._release_service(service_id)


__Task__ = BuildMarketServiceConfig
