import logging
import os
import shutil
import tarfile

from sandbox import sdk2
from sandbox.projects.common.nanny import nanny
from sandbox.projects import resource_types
from sandbox.sandboxsdk.ssh import Key
from sandbox.sdk2.vcs.git import Git


class BuildMarketStaticFiles(nanny.ReleaseToNannyTask2, sdk2.Task):
    """A task that packs static files into tarball"""

    class Parameters(sdk2.Task.Parameters):
        url = sdk2.parameters.String('Git repository url', required=True)
        branch = sdk2.parameters.String('Git repository branch', default='master', required=False)
        path = sdk2.parameters.String('Git repository relative path to pack', default='', required=False)

        with sdk2.parameters.String('Resource type', multiline=True, required=True) as resource_type:
            for rt in resource_types.AbstractResource:
                if 'MARKET' in rt.name:
                    resource_type.values[rt.name] = rt.name

        startrek_ticket = sdk2.parameters.List('Startrek ticket', required=True)

        secret_ssh = sdk2.parameters.String('Vault secret with SSH key', default='robot-market-packer-ssh',
                                            required=True)
        secret_owner = sdk2.parameters.String('Vault owner of the secret', default='MARKETSRE', required=True)

    def _clone_repository(self, checkout_path):
        logging.info('Getting ssh public key and cloning repository {}:{}'.format(
            self.Parameters.url, self.Parameters.branch,
        ))
        with Key(self, self.Parameters.secret_owner, self.Parameters.secret_ssh):
            git = Git(self.Parameters.url)
            git.clone(checkout_path, self.Parameters.branch)

        for ignore_dir in '.git', '.gitignore', 'debian':
            logging.info('Removing ' + ignore_dir)
            shutil.rmtree(os.path.join(checkout_path, ignore_dir), ignore_errors=True)

    def on_execute(self):
        checkout_path = os.path.basename(self.Parameters.url)
        self._clone_repository(checkout_path)
        if self.Parameters.path:
            checkout_path = os.path.join(checkout_path, self.Parameters.path.strip())
        logging.info('Target path ' + checkout_path)
        logging.info('Creating resource')
        resource_type = resource_types.AbstractResource[self.Parameters.resource_type]
        if not hasattr(resource_type, "path_"):
            resource_type.path_ = resource_type.name.lower().replace('_', '-') + ".tar.gz"
        if not hasattr(resource_type, "description_"):
            resource_type.description_ = resource_type.name.lower() + " from " + self.Parameters.url

        data = sdk2.ResourceData(resource_type(self, resource_type.description_, resource_type.path_))

        logging.info('Creating tarball {}'.format(resource_type))
        target_name = os.path.join(resource_type.path_, checkout_path.split('/')[-1])
        with tarfile.open(name=resource_type.path_, mode='w:gz') as tar:
            tar.add(checkout_path, arcname=target_name)

        logging.info('Marking resource as ready')
        data.ready()
