# -*- coding: utf-8 -*-
"""This task creates gencfg groups via utils/market/find_hosts_and_create_group.py"""

import os
import logging

from sandbox import common
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.vcs.svn import Svn
from sandbox.projects.common import utils
from sandbox.projects.common.gencfg.task import IGencfgBuildTask
from contextlib import contextmanager


@contextmanager
def _chdir(path):
    curdir = os.path.abspath(os.curdir)
    os.chdir(path)
    yield
    os.chdir(curdir)


class Args(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'args'
    description = 'Arguments to run the script'
    default_value = []


class CreateGencfgGroup(IGencfgBuildTask):
    type = 'CREATE_GENCFG_GROUP'

    input_parameters = [Args]

    def on_execute(self):
        self._prepare_env()
        self._create_groups()
        self._run_checks()
        self._commit_changes()

    def _prepare_env(self):
        logging.info('Preparing an environment for GenCfg')
        self.clone_and_install_generator(precalc_caches=True, load_cache_resource=True)

    def _create_groups(self):
        logging.info('Creating groups')

        count = 1
        failed = False

        for args in utils.get_or_default(self.ctx, Args):
            process = run_process(
                ' '.join(['/skynet/python/bin/python', './utils/market/find_hosts_and_create_group.py', args]),
                work_dir=self.get_gencfg_path(),
                log_prefix='find_hosts_and_create_group{}'.format(count),
                check=False,
            )

            if process.returncode:
                failed = True
                self.set_info(process.saved_cmd)
                with open(process.stdout_path) as fp:
                    self.set_info(fp.read(), do_escape=False)

        if failed:
            raise common.errors.TaskFailure('Failed to allocate resource for new groups')

    def _run_checks(self):
        logging.info('Running checks')
        self.build_generator(create_resources=False, run_only_checks=True)

    def _check_exists(self, args):
        process = run_process(
            ' '.join(['/skynet/python/bin/python', './utils/common/has_group.py', args.group]),
            work_dir=self.get_gencfg_path(),
            log_prefix='check_group_exists',
            check=False,
        )

        if process.returncode:
            failed = True
            self.set_info(process.saved_cmd)
            with open(process.stdout_path) as fp:
                self.set_info(fp.read(), do_escape=False)

    def _commit_changes(self):
        logging.info('Commiting changes')

        groups_path = 'db'

        with _chdir(os.path.join(self.get_gencfg_path(), groups_path)):
            logging.info('Svn status: %s', Svn.status(path='.'))
            output = Svn.commit(path='.', message='Create new groups via {}'.format(self.type))
            if not output:
                raise common.errors.TaskFailure('Nothing has been commited')

        logging.info('Changes have been commited')


__Task__ = CreateGencfgGroup
