# coding=utf-8

import logging
import os
import os.path
import shutil
import tarfile
from contextlib import contextmanager

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm

from sandbox import sdk2
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.common.ya_deploy import release_integration


@contextmanager
def _chdir(path):
    curdir = os.path.abspath(os.curdir)
    os.chdir(path)
    yield
    os.chdir(curdir)


class CreateResourceFromPlaneGit(release_integration.ReleaseToYaDeployTask2, sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        repository = sdk2.parameters.String("Адрес репозитория в формате ssh://git@...", required=True)
        branch = sdk2.parameters.String("Ветка в репозитоии", default="master")
        commit = sdk2.parameters.String("Хэш коммита")
        repo_subdir = sdk2.parameters.String("Подкаталог в репозитории для публикации в ресурс")
        destination_path = sdk2.parameters.String("Путь к данным относительно корня")
        resource_type = sdk2.parameters.String("Тип ресурса", default="OTHER_RESOURCE", required=True)

        yav_secret = sdk2.parameters.YavSecret("Секрет в Yav", required=True)
        yav_secret_key = sdk2.parameters.String(
            "Ключ секрета с приватным ключом ssh", default="ssh-private", required=True)
        release_to_deploy = sdk2.parameters.Bool("Публиковать релизы в Y.Deploy", default=False)

    def on_execute(self):
        ssh_key = self._get_ssh_key()
        repo_path = self._clone_repo(
            self.Parameters.repository, self.Parameters.branch, self.Parameters.commit, ssh_key)
        if self.Parameters.repo_subdir:
            repo_path = os.path.join(repo_path, self.Parameters.repo_subdir)
        resource_path = self._build_resource(
            repo_path, self.Parameters.destination_path, self.Parameters.resource_type.lower())
        self._create_resource(sdk2.Resource[self.Parameters.resource_type], resource_path)

    def _get_ssh_key(self):
        return self.Parameters.yav_secret.data()[self.Parameters.yav_secret_key]

    def _clone_repo(self, repository, branch, commit, ssh_key):
        logging.info("Clone git repository {repository}".format(repository=repository))
        work_dir_name = "repository"
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(repository)
            git.clone(target_dir=work_dir_name, branch=branch, commit=commit)
        return os.path.join(os.getcwd(), work_dir_name)

    def _build_resource(self, repo_path, destination_path, resource_name):
        resource_destination_path = resource_name + destination_path
        logging.info(resource_destination_path)

        shutil.copytree(src=repo_path, dst=resource_destination_path,
            ignore=shutil.ignore_patterns(".*"))
        logging.info(os.listdir(resource_destination_path))

        resource_tar = resource_name + ".tar.gz"
        with tarfile.open(resource_tar, "w:gz") as tar:
            with _chdir(resource_name):
                for item in os.listdir("."):
                    tar.add(item)

        return resource_tar

    def _create_resource(self, resource_type, resource_path):
        resource = resource_type(self, description="Git tarball package", path=resource_path)
        resource_data = sdk2.ResourceData(resource)
        resource_data.ready()
        logging.info('Resource {} created'.format(resource.id))

    def on_release(self, additional_parameters):
        if self.Parameters.release_to_deploy:
            release_integration.ReleaseToYaDeployTask2.on_release(self, additional_parameters)
