# -*- coding: utf-8 -*-

from urllib2 import urlopen
from urllib2 import URLError
import logging
import socket
import json

from sandbox.common.errors import TaskFailure
from sandbox.sdk2 import Attributes
from sandbox.sdk2 import Resource
from sandbox.sdk2 import ResourceData
from sandbox.sdk2 import Task
from sandbox.sdk2 import parameters


class MarketSreDebPackage(Resource):
    package_name = Attributes.String("Package name", required=True)
    package_version = Attributes.String("Package version", required=True)
    package_branch = Attributes.String("Package branch")
    auto_backup = True


class CreateResourceWithDebPackage(Task):
    """Создание Sandbox ресурса, содержащего deb пакет

    Задача используется для скачиивания из Cacum указанного deb пакета и
    сохранени его как ресурса.
    """

    HTTP_RETRY_COUNT = 5

    CONDUCTOR_URL_PATTERN = "https://c.yandex-team.ru/api/package_version/{}?branch={}"
    DIST_DOWNLOAD_URL_PATTERN = "http://dist.yandex.ru/find?repo={}&pkg={}&ver={}&withurls=true&strict=true"

    class Parameters(Task.Parameters):
        package_name = parameters.String("Package name", required=True)
        is_latest_version = parameters.Bool("Get current version from Conductor", default=True)
        with is_latest_version.value[False]:
            package_version = parameters.String("Package version")
        with is_latest_version.value[True]:
            with parameters.String("Package branch", multiline=True) as package_branch:
                package_branch.values.stable = package_branch.Value("stable", default=True)
                package_branch.values.prestable = package_branch.Value("prestable")
                package_branch.values.testing = package_branch.Value("testing")
                package_branch.values.unstable = package_branch.Value("unstable")
        repository_name = parameters.String("Cacus repository name", required=True)
        duplicate_if_exist = parameters.Bool("Create new resource if already exist", default=False)

    def on_execute(self):

        if self.Parameters.is_latest_version:
            version = self.get_current_version_from_conductor(self.Parameters.package_name,
                                                              self.Parameters.package_branch)
        else:
            version = self.Parameters.package_version

        download_url = self.get_download_url(self.Parameters.repository_name, self.Parameters.package_name, version)

        file_name = download_url.split('/')[-1]
        if self.Parameters.duplicate_if_exist or not MarketSreDebPackage.find(
                attrs=dict(package_name=self.Parameters.package_name, package_version=version, package_branch=self.Parameters.package_branch)).first():
            for _ in range(self.HTTP_RETRY_COUNT):
                result = urlopen(download_url)
                if result.getcode() != 200:
                    continue

                app_res = ResourceData(MarketSreDebPackage(
                    self, file_name, file_name, package_name=self.Parameters.package_name, package_version=version, package_branch=self.Parameters.package_branch
                ))
                app_res.path.write_bytes(result.read())
                app_res.ready()
                return

            raise TaskFailure("Failed to download DEB package")
        else:
            self.set_info("Package already present in Sandbox resources. "
                          "Use \"{}\" input parameter to create another one.".format(
                            CreateResourceWithDebPackage.Parameters.duplicate_if_exist.description))

    def get_current_version_from_conductor(self, name, branch):
        version = None
        for _ in range(self.HTTP_RETRY_COUNT):
            try:
                result = urlopen(self.CONDUCTOR_URL_PATTERN.format(name, branch))
            except (socket.timeout, URLError) as e:
                logging.error("Failure while accessing Conductor: %s", e)
                continue

            try:
                info = json.loads(result.read())
                version = info[name][branch]["version"]
                break
            except ValueError:
                logging.error("Response from Conductor is not valid")
                continue

        if not version:
            raise TaskFailure("Can't determine package version from conductor.")

        return version

    def get_download_url(self, repository, package, version):
        storage_key = None
        for _ in range(self.HTTP_RETRY_COUNT):
            try:
                response = json.loads(urlopen(self.DIST_DOWNLOAD_URL_PATTERN.format(repository, package, version))
                                      .read())
                if not response.get("success", False) or not response["result"]:
                    logging.error("Response from Cacus is not valid")
                    continue

                storage_key = self.get_storage_key_from_response(response["result"], package)
                break
            except (socket.timeout, URLError) as e:
                logging.error("Failure while accessing Cacus: %s", e)
                continue
            except ValueError:
                logging.error("Response from Cucm is not valid")
                continue

        if not storage_key:
            raise TaskFailure("Can't determine package download URL from Cacus(dist)")

        return storage_key

    @staticmethod
    def get_storage_key_from_response(data, package):
        for source in data:
            for deb in source["debs"]:
                if "{}_{}_{}.deb".format(package, source["version"], deb["architecture"]) == deb["package"]:
                    return deb["storage_key"]
