# -*- coding: utf-8 -*-

import logging

from sandbox.projects.market.kgb.BuildMarketGuruAssistantData import BuildMarketGuruAssistantData
from sandbox.projects.market.sre.BuildMarketGuruAssistantAutoWheels import BuildMarketGuruAssistantAutoWheels
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.parameters import SandboxBoolGroupParameter


class ReleaseService(SandboxBoolGroupParameter):
    name = 'release_service'
    choices = [
        ('testing_sas', 'testing_market_guruassistant_sas'),
        ('testing_vla', 'testing_market_guruassistant_vla'),
        ('prod_sas', 'production_market_guruassistant_sas'),
        ('prod_vla', 'production_market_guruassistant_vla'),
        ('prod_iva', 'production_market_guruassistant_iva'),
    ]
    description = 'Release service'
    default_value = 'testing'


class DeployMarketGuruAssistantData(task.SandboxTask):
    """
    This task builds MARKET_GURU_ASSISTANT_DATA and starts deploying Market Guru Assistant's services
    testing_guruassistant and production_guruassistant.

    The task runs a task BUILD_MARKET_GURU_ASSISTANT_DATA as a subtask and waits when it finishes.
    Then it creates Nanny client and sends a resource update request. Nanny commits the a new snapshot
    with the new resource ids from the request and activates it.

    The task BUILD_MARKET_GURU_ASSISTANT_DATA collects pieces of data from other resources
    and then build them into a resource BUILD_MARKET_GURU_ASSISTANT_DATA. Some resource are uploaded by indexers
    and some are fetched via http.
    """

    type = 'DEPLOY_MARKET_GURU_ASSISTANT_DATA'

    input_parameters = [ReleaseService]

    @property
    def service_ids(self):
        return self.ctx.get(ReleaseService.name, '').split()

    @staticmethod
    def _create_link(url):
        return "<a href={0}>{0}</a>".format(url)

    @staticmethod
    def _create_link_on_nanny(service_id):
        return DeployMarketGuruAssistantData._create_link(
            'https://nanny.yandex-team.ru/ui/#/services/catalog/{}/'.format(service_id))

    def _run_and_wait(self, string_id, task_class):
        if self.ctx.get(string_id):
            return True
        child = self.create_subtask(
            task_type=task_class.type,
            description='Auto release of services {}'.format(', '.join(self.service_ids)),
            inherit_notifications=True,
        )
        self.ctx[string_id] = child.id
        utils.wait_all_subtasks_stop()
        utils.check_subtasks_fails(fail_on_failed_children=True)
        return False

    def _deploy_data(self):
        assistant_data_task = channel.sandbox.get_task(self.ctx.get('assistant_data_task'))

        nanny_client = nanny.NannyClient(api_url='http://nanny.yandex-team.ru/',
                                         oauth_token=self.get_vault_data('MARKETDEPLOY', 'nanny_token'))

        for s in self.service_ids:
            nanny_client.update_service_sandbox_resources(s, assistant_data_task.type,
                                                          str(assistant_data_task.id),
                                                          deploy=True,
                                                          recipe='prepare_and_activate')
            self.set_info(self._create_link_on_nanny(s), do_escape=False)

    def on_execute(self):
        logging.info(self.ctx)

        # get auto.ru wheels data, and wait for it
        if self._run_and_wait('wheels_data_task', BuildMarketGuruAssistantAutoWheels):
            # build guruassistant's data with foregoing, and wait for it
            if self._run_and_wait('assistant_data_task', BuildMarketGuruAssistantData):
                self._deploy_data()


__Task__ = DeployMarketGuruAssistantData
