# -*- coding: utf-8 -*-

import logging

from sandbox.projects import DeployNannyDashboard
from sandbox.projects.market.sre import GetMarketResources
from sandbox.projects.common import utils
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task


class DataPreparationTask(parameters.SandboxStringParameter):
    name = 'data_preparation_task'
    description = 'Data preparation task'
    choices = [
        (GetMarketResources.GetMarketResources.type, GetMarketResources.GetMarketResources.type),
    ]
    sub_fields = {
        GetMarketResources.GetMarketResources.type: [GetMarketResources.ResourceList.name],
    }
    required = True


class DeployTask(parameters.SandboxStringParameter):
    name = 'deploy_task'
    description = 'Deploy task'
    choices = [
        (DeployNannyDashboard.DeployNannyDashboard.type, DeployNannyDashboard.DeployNannyDashboard.type),
    ]
    sub_fields = {
        DeployNannyDashboard.DeployNannyDashboard.type: [
            DeployNannyDashboard.ReleaseTask.name,
            DeployNannyDashboard.SandboxReleaseType.name,
            DeployNannyDashboard.NannyDashboardName.name,
            DeployNannyDashboard.NannyDashboardFilter.name,
            DeployNannyDashboard.NannyDashboardRecipeName.name,
        ],
    }
    required = True


class ReleaseTask(DeployNannyDashboard.ReleaseTask):
    required = False


class DeployMarketService(task.SandboxTask):
    type = 'DEPLOY_MARKET_SERVICE'

    input_parameters = [
        DataPreparationTask,
        GetMarketResources.ResourceList,
        DeployTask,
        ReleaseTask,
        DeployNannyDashboard.SandboxReleaseType,
        DeployNannyDashboard.NannyDashboardName,
        DeployNannyDashboard.NannyDashboardFilter,
        DeployNannyDashboard.NannyDashboardRecipeName,
    ]

    subtask_input_parameters = {
        GetMarketResources.GetMarketResources.type: {
            GetMarketResources.ResourcesTTL.name: '14',
        },
        DeployNannyDashboard.DeployNannyDashboard.type: {
            DeployNannyDashboard.VaultOwner.name: 'MARKETDEPLOY',
            DeployNannyDashboard.VaultName.name: 'nanny_token',
            DeployNannyDashboard.NannyWaitDeployParameter.name: True,
        },
    }

    def _create_subtask(self, subtask_type, description):
        input_parameters = self.ctx.copy()
        input_parameters.update(self.subtask_input_parameters[subtask_type])
        subtask = self.create_subtask(
            task_type=subtask_type,
            description=description,
            inherit_notifications=True,
            input_parameters=input_parameters,
        )
        return subtask.id

    def on_execute(self):
        logging.info(self.ctx)

        if not self.ctx.get(DeployNannyDashboard.ReleaseTask.name):
            subtask_type = utils.get_or_default(self.ctx, DataPreparationTask)
            if subtask_type:
                description = 'Get resources for {}'.format(self.type)
                self.ctx[DeployNannyDashboard.ReleaseTask.name] = self._create_subtask(subtask_type, description)
                utils.wait_all_subtasks_stop()
        elif not self.ctx.get(DeployTask.name + '_id'):
            utils.check_subtasks_fails(fail_on_failed_children=True)
            description = 'Deploy resources {}'.format(
                ', '.join(['{}:{}'.format(v, k) for k, v in
                           utils.get_or_default(self.ctx, GetMarketResources.ResourceList).iteritems()]))
            subtask_type = utils.get_or_default(self.ctx, DeployTask)
            if subtask_type:
                self.ctx[DeployTask.name + '_id'] = self._create_subtask(subtask_type, description)
                utils.wait_all_subtasks_stop()
        else:
            utils.check_subtasks_fails(fail_on_failed_children=True)


__Task__ = DeployMarketService
