# -*- coding: utf-8 -*-

import logging
import socket
import yaml

import sandbox.projects.common.yql as yql
from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.projects.common.juggler import jclient
from sandbox.sandboxsdk import environments


class FslbFalsePositivePings(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = [
            environments.PipEnvironment('yandex-yt'),
            environments.PipEnvironment("yql")
        ]

    class Parameters(sdk2.Task.Parameters):
        database = sdk2.parameters.String("YT Database name", required=True,
                                          default="hahn")
        yt_path = sdk2.parameters.String("YT Path to tables with logs", required=True)
        secret = sdk2.parameters.YavSecret("Yav secret with OAuth tokens", required=True)
        juggler_service = sdk2.parameters.String("Juggler service name", required=True,
                                                 default="fslb_false_positive_pings")

    def on_execute(self):
        secret = self.Parameters.secret
        yql_token = secret.data()["YQL_TOKEN"]
        yt_token = secret.data()["YT_TOKEN"]

        yt_table_path = self._get_yt_table_path(yt_token)
        result = self.get_bad_hosts(yql_token, yt_table_path)
        logging.info(str(result))
        self.send_to_juggler(result)

    def _get_yt_table_path(self, yt_token):
        import yt.wrapper as yt

        yt.config["proxy"]["url"] = self.Parameters.database
        yt.config["token"] = yt_token

        tables = yt.list(self.Parameters.yt_path)
        return "/".join((self.Parameters.yt_path, max(tables)))

    def get_bad_hosts(self, yql_token, table_name):
        blacklist = ["::1", "127.0.0.1"]
        query = """
            SELECT DISTINCT
                `x-balancer-ip`,
                `host`
            FROM `{}`
            WHERE
                `x-balancer-port` == 443 AND
                `service_name` == "ping" AND
                `host` != "localhost";""".format(table_name)

        request = yql.run_query(
            query_id="1",
            query=query,
            yql_token=yql_token,
            db=self.Parameters.database,
            wait=False
        )
        results = request.get_results()

        results_list = list()
        if results.is_success:
            for table in results:
                for row in table.rows:
                    ip, host = row
                    if ip not in blacklist:
                        results_list.append({"ip": ip.encode("UTF-8"), "host": host.encode("UTF-8")})
        else:
            for error in results.errors:
                results_list.append(str(error))
            raise TaskError(", ".join(results_list))
        return results_list

    def format_result(self, result_list):
        result = dict()
        for item in result_list:
            vs = socket.gethostbyaddr(item["ip"])[0]
            if vs in result:
                result[vs].append(item)
            else:
                result[vs] = [item]
        return result

    def send_to_juggler(self, result):
        result_by_vs = self.format_result(result)

        if result_by_vs:
            status = "CRIT"
        else:
            status = "OK"

        result_by_vs["META"] = {"META": status}

        for vs in result_by_vs:
            jclient.send_events_to_juggler(
                host=vs,
                service=self.Parameters.juggler_service,
                status=status,
                description=yaml.dump(result_by_vs[vs])
            )
