# -*- coding: utf-8 -*-

import hashlib
import logging
import os

import requests
from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.nanny import nanny
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task

RESOURCE_URL = "http://marketindexer.http.yandex.net:33131/yandex/market-data-getter/lib/{service}/recent/{resource}"
BUFFSIZE = 4096


def split_resource_url(resource_url):
    if resource_url.endswith('/'):
        raise errors.SandboxTaskFailureError('Resource name must be a file not a directory')
    return resource_url.split(':', 1)


def match_checksum(filename, checksum):
    hash = hashlib.md5()
    with open(filename, 'rb') as f:
        while True:
            buf = f.read(BUFFSIZE)
            if not buf:
                break
            hash.update(buf)

    return checksum == hash.hexdigest()


def get_file_from_url(url, filename):
    d = os.path.dirname(filename)
    if not os.path.exists(d):
        logging.info('Create directory {}'.format(d))
        os.makedirs(d)

    logging.info('Fetch file {} from url {}'.format(filename, url))
    r = requests.get(url, stream=True)
    r.raise_for_status()
    with open(filename, 'wb') as f:
        for chunk in r.iter_content(chunk_size=BUFFSIZE):
            f.write(chunk)


def get_checksum(resource_url):
    checksum_file = 'MD5SUMS'
    service_name, resource_name = split_resource_url(resource_url)
    url = RESOURCE_URL.format(service=service_name, resource=checksum_file)
    filename = os.path.join('checksums', service_name, checksum_file)

    if not os.path.exists(filename):
        try:
            get_file_from_url(url, filename)
        except requests.HTTPError as e:
            logging.error('Failed to fetch file {} from url {}: {}'.format(filename, url, e))
            return

    with open(filename) as f:
        for l in f.readlines():
            c, f = l.split()
            if resource_name == f:
                return c


def get_market_resource(resource_url):
    service_name, resource_name = split_resource_url(resource_url)
    url = RESOURCE_URL.format(service=service_name, resource=resource_name)
    filename = os.path.join(service_name, resource_name)

    get_file_from_url(url, filename)

    return filename


class ResourceList(parameters.DictRepeater, parameters.SandboxStringParameter):
    name = 'resource_list'
    description = 'Market getter resource (type:filename)'
    choices = sorted(((rt.name, rt.name) for rt in resource_types.AbstractResource if 'MARKET' in rt.name))
    required = True
    default_value = ''


class ResourcesTTL(parameters.SandboxStringParameter):
    name = 'resources_ttl'
    description = 'Resources TTL'
    default_value = 'inf'


class GetMarketResources(task.SandboxTask, nanny.ReleaseToNannyTask):
    type = 'GET_MARKET_RESOURCES'

    input_parameters = [ResourceList, ResourcesTTL]

    def on_execute(self):
        for resource, resource_type in utils.get_or_default(self.ctx, ResourceList).iteritems():
            filename = get_market_resource(resource)
            checksum = get_checksum(resource)

            if checksum and not match_checksum(filename, checksum):
                raise errors.SandboxTaskFailureError('Checksum mismatch')

            r = self.create_resource(self.descr, filename, resource_type)
            self.mark_resource_ready(r.id)

    def on_release(self, additional_parameters):
        ttl = utils.get_or_default(self.ctx, ResourcesTTL)

        nanny.ReleaseToNannyTask.on_release(self, additional_parameters)

        # Copied from yasandbox.proxy.task.Task
        logging.debug("Release parameters: %r", additional_parameters)
        self.send_release_info_to_email(additional_parameters)
        self.mark_released_resources(additional_parameters["release_status"], ttl)


__Task__ = GetMarketResources
