# -*- coding: utf-8 -*-

import os
import logging
from sandbox import sdk2
import sandbox.common.types.client as ctc
from sandbox import common
from tempfile import mkdtemp
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects import resource_types
from sandbox.sandboxsdk.ssh import Key
from sandbox.sdk2.vcs.git import Git


VAULT_SSH = "csadmin-ssh"
VAULT_OWNER = "MARKETSRE"


class GitToEtcd(sdk2.Task):
    """
    Export datasources from git to etcd via update_datasources tool
    """

    class Requirements(sdk2.Task.Requirements):
        client_tags = ctc.Tag.LINUX_TRUSTY

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        with sdk2.parameters.String("etcd cluster", required=True) as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["stable"] = "stable"

        repo = sdk2.parameters.String("Git repository", default="git@github.yandex-team.ru:cs-admin/datasources-ng.git", required=True)
        branch = sdk2.parameters.String("Git branch to clone", default="master", required=True)
        username = sdk2.parameters.String("etcd username", default="datasources_readwrite_all", required=True)
        vault_key = sdk2.parameters.String("Vault key, containing etcd password", default="market_etcd_datasources_readwrite_all", required=True)
        force = sdk2.parameters.Bool("DANGEROUS: Force push datasources")

        up_ds = sdk2.parameters.Resource(
            "update_datasources binary",
            resource_type=resource_types.UPDATE_DATASOURCES_BIN,
            required=True
        )

    def _git_clone(self, repo, branch):
        logging.info("Cloning repo {}, branch {}".format(repo, branch))
        with Key(self, VAULT_OWNER, VAULT_SSH):
            dirname = mkdtemp()
            git = Git(repo)
            git.clone(dirname, branch)
            return dirname

    def on_execute(self):
        up_ds = str(sdk2.ResourceData(self.Parameters.up_ds).path)

        repo = self.Parameters.repo
        br = self.Parameters.branch
        username = self.Parameters.username
        environment = self.Parameters.environment

        endpoints = {}
        if environment in ["testing"]:
            endpoints = {
                "tst_cross": "etcd.tst.vs.market.yandex.net:3379",
            }
        elif environment in ["stable"]:
            endpoints = {
                "cross": "etcd.vs.market.yandex.net:3379",
            }

        run_env = os.environ.copy()
        run_env["ETCD_PASSWORD"] = sdk2.Vault.data(self.Parameters.vault_key)

        dirname = self._git_clone(repo, br)
        for prefix in [f for f in os.listdir(dirname) if not f.startswith(".") and os.path.isdir(os.path.join(dirname, f))]:
            logging.info("Exporting dir {}".format(prefix))
            for dc, ep in endpoints.iteritems():
                logging.info("Working with {} datacenter".format(dc))
                with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("update_datasources")) as pl:
                    if self.Parameters.force:
                        pr = sp.Popen("{} -e {} -u {} -p $ETCD_PASSWORD -f {} -b {} -F".format(up_ds, ep, username, os.path.join(dirname, prefix), prefix), shell=True, stdout=pl.stdout, stderr=pl.stdout, env=run_env).wait()
                    else:
                        pr = sp.Popen("{} -e {} -u {} -p $ETCD_PASSWORD -f {} -b {}".format(up_ds, ep, username, os.path.join(dirname, prefix), prefix), shell=True, stdout=pl.stdout, stderr=pl.stdout, env=run_env).wait()
                    pl.logger.propagate = 1
                    if pr != 0:
                        raise common.errors.TaskError("Something went wrong for dc {}, dir {}".format(dc, prefix))
