# coding=utf-8

import logging
import os
import os.path
import re

from collections import OrderedDict
from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia, SvnError


class MarketAddServiceToYamlConfigs(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        arcadia_user = "zomb-sandbox-rw"
        configs_dir = sdk2.parameters.String(
            "Путь к каталогу с конфигами относительно корня Аркадии", required=True)
        existing_service = sdk2.parameters.String(
            "Существующий сервис для поиска нужных конфигов", required=True)
        new_service = sdk2.parameters.String(
            "Новый сервис, который будем добавлять", required=True)
        startrek_ticket = sdk2.parameters.String(
            "Тикет в Стартреке", required=False)
        with sdk2.parameters.Output:
            commit_url = sdk2.parameters.String("Commit URL", default="", required=True)

    def on_execute(self):
        configs_dir = self._checkout_configs_dir()
        configs = self._find_configs(configs_dir)
        logging.critical(str(configs))
        for config_path, config_data in configs.items():
            self._get_payload_for_config(config_data)
            self._add_service_to_config(config_path, config_data)
        if configs:
            self._commit_changes(configs_dir)
        else:
            logging.info("Nothing to commit")
            self.Parameters.commit_url = "Not changed"

    def _checkout_configs_dir(self):
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, str(self.Parameters.configs_dir))
        configs_dir = Arcadia.checkout(arcadia_url, str(self.path("configs_dir")))
        logging.info("Arcadia is checkouted to {}".format(configs_dir))
        return configs_dir

    def _check_service_in_config(self, content):
        indexes = OrderedDict()
        for idx, line in enumerate(content):
            if self.Parameters.new_service in line:
                raise ValueError("New service found in config")
            if self.Parameters.existing_service in line:
                indexes[idx] = []
        return indexes

    def _find_configs(self, configs_path):
        logging.info("Searching configs with service")
        configs = {}
        for file_name in os.listdir(configs_path):
            file_path = os.path.join(configs_path, file_name)
            if os.path.isfile(file_path) and (file_path.endswith(".yaml") or
                                              file_path.endswith(".yml")):
                with open(file_path) as fd:
                    config_content = fd.readlines()
                    try:
                        indexes = self._check_service_in_config(config_content)
                    except ValueError:
                        logging.warning("New service found in config %s", file_name)
                    if indexes:
                        configs[file_path] = {"indexes": indexes, "content": config_content}
        logging.info("Found configs: %s", ", ".join(configs))
        return configs

    def _add_service_to_config(self, config_path, config_data):
        logging.info("Adding service to config %s", config_path)
        idx_offset = 0
        for index in config_data["indexes"]:
            idx = index + idx_offset
            original_string = config_data["content"][idx]
            new_string = original_string.replace(
                str(self.Parameters.existing_service), str(self.Parameters.new_service))
            config_data["content"].insert(idx, new_string)
            idx_offset += 1

            payload = config_data["indexes"][index]
            payload_offset = 1
            for line in payload:
                config_data["content"].insert(idx + payload_offset, line)
                payload_offset += 1
                idx_offset += 1

        with open(config_path, "w") as fd_w:
            fd_w.writelines(config_data["content"])

    def _get_payload_for_config(self, config_data):
        logging.info("Get payload for configs")
        for idx in config_data["indexes"]:
            offset = 1
            while (idx + offset) <= len(config_data["content"]) - 1:
                first_string = config_data["content"][idx]
                current_string = config_data["content"][idx + offset]
                if "sort_order: " in current_string:
                    break
                if current_string.lstrip().startswith("-"):
                    break
                # Проверка на выход из блока. Проверяем, что в первой строке отступ больше, чем в текущей
                if first_string.find(first_string.lstrip()) >= current_string.find(current_string.lstrip()):
                        break
                config_data["indexes"][idx].append(current_string)
                offset += 1
        logging.critical(config_data["indexes"])

    def _commit_changes(self, configs_dir):
        logging.info("Commiting changes to Arcadia")
        logging.info('Svn status: %s', Arcadia.status(path=configs_dir))
        logging.info('Svn diff: \n%s', Arcadia.diff(configs_dir))

        try:
            if self.Parameters.startrek_ticket:
                ticket = self.Parameters.startrek_ticket + " "
            else:
                ticket = ""
            result = Arcadia.commit(path=configs_dir,
                                    message="{}Add new service {} to config".format(ticket, self.Parameters.new_service),
                                    user=self.Parameters.arcadia_user)
            logging.info("Commit result: %s", result)
            revision = re.findall(r'Committed revision (\d+)\.', result)
            if revision:
                self.Parameters.commit_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            logging.error(e.message)
            # For Arcadia reviews
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            if review:
                self.Parameters.commit_url = "https://a.yandex-team.ru/review/{}".format(review[0])
