# coding=utf-8

import logging
import os
import os.path
import tarfile
import tempfile

import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox.projects.market.resources.sre import MARKET_ALERTS_CONFIGURATOR
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia


class MarketAlertsConfiguratorExec(sdk2.Task):
    """Настройка проверок Juggler с помощью market-alerts-configurator."""

    CONFIGURATOR_BIN = 'market-alerts-configurator'
    YAV_SECRET = 'sec-01cyha56hmq89wv6rc8g6ae9dz'

    class Parameters(sdk2.Task.Parameters):
        arcadia_conf_path = sdk2.parameters.String(
            'Путь к конфигам относительно корня Аркадии',
            default='market/sre/conf/market-alerts-configs/configs'
        )
        arcadia_global_conf_path = sdk2.parameters.String(
            'Путь к глобальным конфигам относительно корня Аркадии',
            default='market/sre/conf/market-alerts-configs/globals'
        )
        cleanup_tag = sdk2.parameters.String(
            'Тег по которому находят все проверки, которые должны контролироваться market-alerts-configurator',
            default='_market_cleanup_tag_'
        )
        mandatory_tags = sdk2.parameters.String(
            'Обязательные теги, добавляемые всем проверкам. Записываются в строку через запятую',
            default='market,_market_'
        )
        market_alerts_configurator_resource = sdk2.parameters.LastReleasedResource(
            'Executable market-alerts-configurator',
            resource_type=MARKET_ALERTS_CONFIGURATOR,
            state=ctr.State.READY,
            attrs={'released': 'unstable'},
            required=True
        )

    def on_execute(self):
        tempdir = tempfile.gettempdir()

        # Extract MARKET_ALERTS_CONFIGURATOR
        resource_data = sdk2.ResourceData(self.Parameters.market_alerts_configurator_resource)
        with tarfile.open(str(resource_data.path), 'r:gz') as tar:
            tar.extractall(path=tempdir)

        # Get configs
        arcadia_conf_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.Parameters.arcadia_conf_path)
        configs_path = Arcadia.export(url=arcadia_conf_url, path=os.path.join(tempdir, 'configs'))
        arcadia_global_conf_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.Parameters.arcadia_global_conf_path)
        global_configs_path = Arcadia.export(url=arcadia_global_conf_url, path=os.path.join(tempdir, 'globals'))

        # Execute market-alerts-configurator
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(self.CONFIGURATOR_BIN)) as pl:
            secret = sdk2.yav.Secret(self.YAV_SECRET)
            command_env = os.environ.copy()
            command_env['JUGGLER_TOKEN'] = secret.data()['JUGGLER_TOKEN']
            command = [os.path.join(tempdir, self.CONFIGURATOR_BIN),
                       '--conf-dir', configs_path,
                       '--globals-dir', global_configs_path,
                       '--cleanup-tag', self.Parameters.cleanup_tag,
                       '--mandatory-tags', self.Parameters.mandatory_tags,
                       '--verbose']
            process = sp.Popen(command, stdout=pl.stdout, stderr=sp.STDOUT, env=command_env)
            process.wait()
            pl.raise_for_status(process)
