# coding=utf-8
import logging
import os
import re
import yaml

from sandbox import sdk2
from sandbox.sdk2.vcs.svn import Arcadia, SvnError


class MarketBalancerConfigCommit(sdk2.Task):
    """Commit new Balancer Config to Arcadia. For Market balancers only.
    """

    class Parameters(sdk2.Task.Parameters):
        # yandex-balancer | nginx
        startrek_ticket = sdk2.parameters.String("Startrek ticket", default=None)
        top_balancer_type = sdk2.parameters.String("Top Balancer type", default="yandex-balancer", required=True)
        env_type = sdk2.parameters.String("Environment", default="test", required=True)        # prod | test | pipeline
        config_name = sdk2.parameters.String("Config name", required=True)
        haproxy_content = sdk2.parameters.String("Haproxy config content", multiline=True, required=False)
        top_balancer_content = sdk2.parameters.String("Top balancer config content", multiline=True, required=False)
        addition_ip_map = sdk2.parameters.String("Addition for ip-service-map", multiline=True, required=False)
        arcadia_user = sdk2.parameters.String("Arcadia user", default="zomb-sandbox-rw")
        with sdk2.parameters.Output:
            arcadia_url = sdk2.parameters.String("Commit URL", default="", required=True)

    def on_execute(self):

        arcadia_conf_path = "arcadia:/arc/trunk/arcadia/market/sre/conf"

        conf_subdirs = {
            "yandex-balancer": ("fslb", "values-available"),
            "nginx": ("slb-nginx", "etc/nginx/values-available"),
            "haproxy": ("slb-haproxy", "etc/haproxy/values-available")
        }

        # Check environment
        if self.Parameters.env_type not in ['prod', 'test', 'pipeline']:
            raise Exception("!! Invalid environment specified")

        # Check filename
        if self.Parameters.config_name and not re.match(r"\d+-[a-z0-9_.-]+.yaml", self.Parameters.config_name):
            raise Exception("Not valid config name")

        # Get startrek_ticket with fallback
        if self.Parameters.startrek_ticket:
            st_ticket = self.Parameters.startrek_ticket
        else:
            st_ticket = "CSADMIN-" + self.Parameters.config_name.split('-')[0]

        logging.info("Starting MarketBalancerConfigCommit task. Top balancer type: {}. Environment: {}".format(
            self.Parameters.top_balancer_type, self.Parameters.env_type))

        # Checkout Arcadia
        arcadia_dir = Arcadia.checkout(url=arcadia_conf_path, path=str(self.path("arcadia")))

        # Set variables
        pipeline_ticket_dir = os.path.join(arcadia_dir, "pipeline", st_ticket)
        pipeline_haproxy_conf_dir = os.path.join(pipeline_ticket_dir, "haproxy")
        pipeline_top_balancer_conf_dir = os.path.join(pipeline_ticket_dir, self.Parameters.top_balancer_type)
        target_haproxy_conf_dir = os.path.join(
                arcadia_dir, conf_subdirs['haproxy'][0], self.Parameters.env_type, conf_subdirs['haproxy'][1])
        target_top_balancer_conf_dir = os.path.join(
                arcadia_dir, conf_subdirs[self.Parameters.top_balancer_type][0],
                self.Parameters.env_type, conf_subdirs[self.Parameters.top_balancer_type][1])
        pipeline_add_ip_map_conf_file = os.path.join(pipeline_top_balancer_conf_dir, "addition-ip-map.yaml")
        target_ip_map_conf_file = os.path.join(arcadia_dir, "fslb/{}/values-static/00-ip-service-map.yaml".format(self.Parameters.env_type))

        haproxy_configs = {}
        top_balancer_configs = {}

        if self.Parameters.env_type == "pipeline":
            # If "pipeline" => create directories and load configs from parameters
            if os.path.isdir(pipeline_ticket_dir):
                raise Exception("Directory {} already exists".format(pipeline_ticket_dir))
            os.makedirs(pipeline_top_balancer_conf_dir)
            os.makedirs(pipeline_haproxy_conf_dir)
            haproxy_dict = yaml.load(self.Parameters.haproxy_content)
            haproxy_config = self.Parameters.haproxy_content
            top_balancer_dict = yaml.load(self.Parameters.top_balancer_content)
            top_balancer_config = self.Parameters.top_balancer_content

            if self.Parameters.addition_ip_map:
                addition_ip_map_dict = yaml.load(self.Parameters.addition_ip_map)
                addition_ip_map_config = self.Parameters.addition_ip_map
            else:
                addition_ip_map_dict = addition_ip_map_config = None
            # We dont need original IP-MAP here
            orig_ip_map_dict = orig_ip_map_config = None

            # New line at the end of configs
            if haproxy_config and haproxy_config[-1] != '\n':
                haproxy_config += '\n'
            if top_balancer_config and top_balancer_config[-1] != '\n':
                top_balancer_config += '\n'
            if addition_ip_map_config and addition_ip_map_config[-1] != '\n':
                addition_ip_map_config += '\n'
        else:
            # We will commit to main directories

            # Load configs from pipeline directory (if they exist)
            # Haproxy
            if os.path.isdir(pipeline_haproxy_conf_dir):
                haproxy_files = os.listdir(pipeline_haproxy_conf_dir)
                for pipeline_haproxy_conf_file in haproxy_files:
                    with open(os.path.join(pipeline_haproxy_conf_dir, pipeline_haproxy_conf_file), 'r') as f:
                        haproxy_config = f.read()
                    # New line at the end of configs
                    if haproxy_config and haproxy_config[-1] != '\n':
                        haproxy_config += '\n'
                    haproxy_dict = yaml.load(haproxy_config)

                    logging.info("Haproxy loaded YAML: {}".format(haproxy_dict))

                    # Firstly check that we will not overwrite existing config files
                    target_haproxy_conf_file = os.path.join(target_haproxy_conf_dir, pipeline_haproxy_conf_file)
                    if os.path.exists(target_haproxy_conf_file):
                        raise Exception("!! File already exists: {}".format(target_haproxy_conf_file))

                    haproxy_configs[pipeline_haproxy_conf_file] = haproxy_config

            # Top balancer
            if os.path.isdir(pipeline_top_balancer_conf_dir):
                top_balancer_files = os.listdir(pipeline_top_balancer_conf_dir)
                for pipeline_top_balancer_conf_file in top_balancer_files:
                    if pipeline_top_balancer_conf_file == "addition-ip-map.yaml":
                        continue
                    with open(os.path.join(pipeline_top_balancer_conf_dir, pipeline_top_balancer_conf_file), 'r') as f:
                        top_balancer_config = f.read()
                    # New line at the end of configs
                    if top_balancer_config and top_balancer_config[-1] != '\n':
                        top_balancer_config += '\n'
                    top_balancer_dict = yaml.load(top_balancer_config)

                    logging.info("Top Balancer loaded YAML: {}".format(top_balancer_dict))

                    # Firstly check that we will not overwrite existing config files
                    target_top_balancer_conf_file = os.path.join(target_top_balancer_conf_dir, pipeline_top_balancer_conf_file)
                    if os.path.exists(target_top_balancer_conf_file):
                        raise Exception("!! File already exists: {}".format(target_top_balancer_conf_file))

                    top_balancer_configs[pipeline_top_balancer_conf_file] = top_balancer_config

            # Yandex-balancer additions
            if self.Parameters.top_balancer_type == "yandex-balancer" and os.path.isfile(pipeline_add_ip_map_conf_file):
                # Load additions for ip-service-map
                with open(pipeline_add_ip_map_conf_file, 'r') as f:
                    addition_ip_map_config = f.read()
                addition_ip_map_dict = yaml.load(addition_ip_map_config)

                # Load current ip-service-map
                with open(target_ip_map_conf_file, 'r') as f:
                    orig_ip_map_config = f.read()
                orig_ip_map_dict = yaml.load(orig_ip_map_config)
            else:
                addition_ip_map_dict = addition_ip_map_config = None
                orig_ip_map_dict = orig_ip_map_config = None

        # New line at the end of configs
        if addition_ip_map_config and addition_ip_map_config[-1] != '\n':
            addition_ip_map_config += '\n'

        if self.Parameters.top_balancer_type == "yandex-balancer":
            logging.info("Additions loaded YAML: {}".format(addition_ip_map_dict))
            if orig_ip_map_dict:
                logging.info("Original IP-MAP config: {}".format(orig_ip_map_dict))

        # Adding configs to Arcadia
        # 1. If env_type is pipelne => pre-commit to pipeline DIR
        if self.Parameters.env_type == "pipeline":
            pipeline_haproxy_conf_file = os.path.join(pipeline_haproxy_conf_dir, self.Parameters.config_name)
            with open(pipeline_haproxy_conf_file, 'w') as f:
                f.write(haproxy_config)
            pipeline_top_balancer_conf_file = os.path.join(pipeline_top_balancer_conf_dir, self.Parameters.config_name)
            with open(pipeline_top_balancer_conf_file, 'w') as f:
                f.write(top_balancer_config)
            if addition_ip_map_config:
                with open(pipeline_add_ip_map_conf_file, 'w') as f:
                    f.write(addition_ip_map_config)
            Arcadia.add(pipeline_ticket_dir)
            commit_message = "{} Added new balancer configs for review SKIP_CHECK".format(st_ticket)
        # 2. Else => commit to main repository
        else:
            for haproxy_file in haproxy_configs.keys():
                target_haproxy_conf_file = os.path.join(target_haproxy_conf_dir, haproxy_file)
                haproxy_config = haproxy_configs[haproxy_file]
                with open(target_haproxy_conf_file, 'w') as f:
                    f.write(haproxy_config)
                Arcadia.add(target_haproxy_conf_file)

            for top_balancer_file in top_balancer_configs.keys():
                target_top_balancer_conf_file = os.path.join(target_top_balancer_conf_dir, top_balancer_file)
                top_balancer_config = top_balancer_configs[top_balancer_file]
                with open(target_top_balancer_conf_file, 'w') as f:
                    f.write(top_balancer_config)
                Arcadia.add(target_top_balancer_conf_file)

            if addition_ip_map_config:
                patching_env = 'production' if self.Parameters.env_type == 'prod' else 'testing'
                # Dictionary merging (to test it vs patched config later)
                addition_vs_dict = addition_ip_map_dict['main'][patching_env]

                # Checking whether dicts contain same keys
                orig_size = len(orig_ip_map_dict['main'][patching_env])
                addition_size = len(addition_vs_dict)
                orig_ip_map_dict['main'][patching_env].update(addition_vs_dict)

                if len(orig_ip_map_dict['main'][patching_env]) != orig_size + addition_size:
                    raise Exception("!! Duplicating values found, can't patch map-ip config")

                # For txt patch we need only strings under `<environment>:` section
                in_env_section = False
                txt_patch = ""
                for line in addition_ip_map_config.split('\n'):
                    match_env = re.match(r' {2}(\w+):', line)
                    if match_env:
                        if match_env.groups()[0] != patching_env:
                            raise Exception("!! Environment in addition ip-map file differs from balancer environment")
                        in_env_section = True
                    elif in_env_section and line:
                        txt_patch += "{}\n".format(line)

                # IP-service-map file patching
                result_addition_ip_map_config = ""
                env_passed = ""
                empty_line_found = False  # Do not produce double empty lines
                patched = False
                for line in orig_ip_map_config.split('\n'):
                    match_env = re.match(r' {2}(\w+):', line)
                    if match_env:  # It is environment name
                        # Check previous environment
                        if env_passed == patching_env:
                            # We need to patch it here!
                            result_addition_ip_map_config += txt_patch
                            patched = True
                        if empty_line_found:
                            result_addition_ip_map_config += '\n'
                            empty_line_found = False
                        env_passed = match_env.groups()[0]
                    if not line:
                        empty_line_found = True
                    else:
                        if empty_line_found:
                            result_addition_ip_map_config += '\n'
                            empty_line_found = False
                        result_addition_ip_map_config += "{}\n".format(line)
                # At the end of original file
                if not patched:
                    if env_passed == patching_env:
                        result_addition_ip_map_config += txt_patch
                    else:
                        raise Exception("!! Wrong environment in addition ip-map file")
                patched_txt_dict = yaml.load(result_addition_ip_map_config)
                if patched_txt_dict != orig_ip_map_dict:
                    raise Exception("!! Something wrong while patching map-ip config")

                with open(target_ip_map_conf_file, 'w') as f:
                    f.write(result_addition_ip_map_config)

            # Remove pipeline's configs
            Arcadia.delete(pipeline_ticket_dir)
            commit_message = "{} Commited new balancer configs".format(st_ticket)

        # logging.info("SVN status:\n{}".format(Arcadia.status(path=arcadia_dir)))
        try:
            result = Arcadia.commit(arcadia_dir, commit_message, self.Parameters.arcadia_user)
            revision = re.findall(r'Committed revision (\d+)\.', result)
            self.Parameters.arcadia_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            # For Arcadia reviews
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            self.Parameters.arcadia_url = "https://a.yandex-team.ru/review/{}".format(review[0])

        logging.info("Commit/Review URL: {}\nAll done".format(self.Parameters.arcadia_url))
