# coding: utf-8
from __future__ import print_function
from __future__ import unicode_literals

import hashlib
import json
import logging
import os
import os.path
import re
from requests.packages.urllib3.util.retry import Retry
from urlparse import urljoin

import requests
import sandbox.common.types.misc as ctm
from lxml import etree
from requests.adapters import HTTPAdapter

from sandbox import sdk2
from sandbox.projects.common.juggler import jclient
from sandbox.common.utils import get_task_link

log = logging.getLogger(__name__)
JUGGLER_HOST = 'market.sandbox'
JUGGLER_SERVICE = 'market_banners_avatars'


class MarketBannersAvatars(sdk2.Task):
    """Upload market banners from Adfox to MDS Avatars storage.

    - Upload banners to avatar storage
    - Update urls in Adfox to point to new banners

    Robot OAuth token to access adfox api are stored in yav market-sre-adfox-robot-credentials
    https://yandex.ru/dev/adfox/doc/v.1/concepts/format-docpage/#format__login

    XXX: Banner urls in Adfox are stored without protocol (e.g. no "https:" part).
    """
    ADFOX_API_URL = 'https://adfox.yandex.ru/api/v1'

    class Requirements(sdk2.Requirements):
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        upload_limit = sdk2.parameters.Integer(
            "Limit the number of banners to be uploaded to this value. 0 - all",
            default=100,
        )
        testing = sdk2.parameters.Bool(
            "Upload to MDS testing servers, don't change urls in Adfox.",
            default=False
        )
        only_upload = sdk2.parameters.Bool(
            "Upload to MDS production servers, don't change urls in Adfox.",
            default=False
        )
        secrets = sdk2.parameters.YavSecret(
            "Yav secret with OAuth tokens"
        )

    def init(self):
        """ Should be called in on_execute """
        secret = self.Parameters.secrets
        self.auth_token = secret.data()["auth_token"]

        self.session = requests.Session()
        retries = Retry(total=5,
                        backoff_factor=0.1,
                        status_forcelist=[500, 502, 503, 504])
        self.session.mount('http://', HTTPAdapter(max_retries=retries))
        self.session.mount('https://', HTTPAdapter(max_retries=retries))

        self.mds_read_url = 'avatars.mds.yandex.net'
        self.mds_write_url = 'http://avatars-int.mds.yandex.net:13000'
        if self.Parameters.testing:
            self.mds_read_url = 'avatars.mdst.yandex.net'
            self.mds_write_url = 'http://avatars-int.mdst.yandex.net:13000'

    def on_execute(self):
        self.init()
        banners_uploaded = -1

        for elem, post_param_name in self.iter_banner_xml_elements():
            if re.match(r'^//banners\.adfox.*?\.(jpe?g|png)$', elem.text) is None:
                continue

            parent = elem.getparent()
            banner_id = parent.find('ID').text
            link = 'https:' + elem.text

            banners_uploaded += 1
            if self.Parameters.upload_limit > 0 and banners_uploaded >= self.Parameters.upload_limit:
                log.info("Uploaded %d banners", banners_uploaded)
                return

            mds_url = self.upload_banner(banner_id, link)

            if not (self.Parameters.testing or self.Parameters.only_upload):
                self.update_banner(banner_id, post_param_name, mds_url)
        log.info("Uploaded %d banners", banners_uploaded)

    def on_success(self, prev_status):
        jclient.send_events_to_juggler(JUGGLER_HOST, JUGGLER_SERVICE, "OK", get_task_link(self.id))
        super(MarketBannersAvatars, self).on_success(prev_status)

    def on_failure(self, prev_status):
        jclient.send_events_to_juggler(JUGGLER_HOST, JUGGLER_SERVICE, "CRIT", get_task_link(self.id))
        super(MarketBannersAvatars, self).on_failure(prev_status)

    def upload_banner(self, banner_id, url):
        """ Upload banner to MDS Avatars storage. Return banner url in MDS.

        If this banner has already been uploaded to MDS, it will return 403 'update is prohibited'.
        In this case add a postfix to the filename. Can happen after 'MarketBannersRollback' has
        been run to reupload the banners with different MDS settings.
        """
        avatar_filename = '{filename}.{banner_id}'.format(
            filename=os.path.basename(url),
            banner_id=banner_id,
        )
        version = 0
        postfix = ''

        while True:
            if version:
                postfix = '.{}'.format(version)
            upload_filename = '{}{}'.format(avatar_filename, postfix)
            upload_url = urljoin(self.mds_write_url, '/put-market_banners/{}'.format(
                upload_filename,
            ))
            response = self.mds_request(
                'POST',
                upload_url,
                params={
                    'url': url,
                },
                # XXX: use 'files' instead of 'data' to send multipart/formdata
                files={
                    'data': json.dumps({'original_url': url}),
                },
            )

            # Add postfix to make banner filename unique in case of the reupload
            if response.status_code == 403 and response.json()['description'] == 'update is prohibited':
                version += 1
                continue

            response.raise_for_status()
            # Successfully uploaded
            break

        data = response.json()
        group_id = data['group-id']

        mds_url = '//{mds_url}/get-market_banners/{group_id}/{filename}/orig'.format(
            mds_url=self.mds_read_url,
            filename=upload_filename,
            group_id=group_id,
        )
        log.info("Uploaded banner: https:%s", mds_url)

        return mds_url

    def mds_request(self, *args, **kwargs):
        response = self.session.request(*args, **kwargs)
        if response.status_code >= 500:
            log.error(
                "Banner upload failed: %s %s %s %s",
                response.status_code, response.request.method, response.request.url, response.reason,
            )

        if response.status_code >= 400:
            data = response.json()
            log.error(
                "Banner upload failed: %s %s %s %s",
                response.status_code, response.request.method, response.request.url, data['description'],
            )
        return response

    def iter_banner_xml_elements(self):
        """ Yields xml elements representing one banner from Adfox answer xml.

        elem - xml element
        post_param_name - name used to update the value of this parameter.
            For "parameterN" equals to "userN".
        """
        pager_limit = 100
        offset = 0
        page = 0
        total_pages = 1

        while page < total_pages:
            response = self.fetch_banners(pager_limit, offset)

            result = response.find('result')
            page = int(result.find('page').text)
            total_pages = int(result.find('total_pages').text)

            for elem in result.iter():
                param_match = re.match(r'^parameter(\d+)$', elem.tag)
                if param_match is None:
                    continue
                if not elem.text:
                    continue

                param_num = param_match.group(1)
                post_param_name = 'user%s' % param_num

                yield elem, post_param_name

            offset += 100

    def fetch_banners(self, limit, offset):
        """ Fetch banner info from Adfox """
        page = self.session.get(
            self.ADFOX_API_URL,
            headers={
                "X-Yandex-API-Key": self.auth_token
            },
            params={
                'object': 'account',
                'action': 'list',
                'actionObject': 'banner',
                'limit': limit,
                'offset': offset,
            }
        )
        page.raise_for_status()
        response = etree.XML(page.content)

        status = int(response.find('status').find('code').text)
        if status != 0:
            print(page.content)
            raise RuntimeError('Failed to fetch banners: {}'.format(page.content.decode('cp1251', errors='replace')))
        return response

    def update_banner(self, banner_id, param_name, url):
        """ Update banner url in Adfox """
        response = self.session.get(
            self.ADFOX_API_URL,
            headers={
                "X-Yandex-API-Key": self.auth_token
            },
            params={
                'object': 'banner',
                'action': 'modify',
                'actionObject': 'banner',
                'objectID': banner_id,
                param_name: url,
            }
        )
        response.raise_for_status()
