# -*- coding: utf-8 -*-

import os
import logging
import yaml
import ConfigParser
import xml.etree.ElementTree as ET
from tempfile import mkdtemp
from sandbox import sdk2
from sandbox import common
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.git import Git
from sandbox.sandboxsdk.ssh import Key


# Datasources repos
VAULT_SSH = "csadmin-ssh"
VAULT_OWNER = "MARKETSRE"
DATASOURCES = "git@github.yandex-team.ru:cs-admin/datasources.git"
DATASOURCES_NG = "git@github.yandex-team.ru:cs-admin/datasources-ng.git"


class MarketConvertDatasources(sdk2.Task):
    """
    Convert market datasources from old formates to standard one
    """

    class Parameters(sdk2.Task.Parameters):
        kill_timeout = 600

        with sdk2.parameters.String("Environment to synchronize", required=True) as environment:
            environment.values["testing"] = environment.Value("testing", default=True)
            environment.values["prestable"] = "prestable"
            environment.values["production"] = "production"

        with sdk2.parameters.Dict("List of pairs datasource file and parser") as MARKET_DATASOURCES:
            MARKET_DATASOURCES.values["_java_properties"] = MARKET_DATASOURCES.Value("Java properties")
            MARKET_DATASOURCES.values["_plain_text"] = MARKET_DATASOURCES.Value("Put content as is")
            MARKET_DATASOURCES.values["_cfg"] = MARKET_DATASOURCES.Value("Config file")
            MARKET_DATASOURCES.values["_datasources_xml"] = MARKET_DATASOURCES.Value("datasources.xml")
            MARKET_DATASOURCES.values["_mbi_endpoints_ent"] = MARKET_DATASOURCES.Value("mbi-endpoints.ent")
            MARKET_DATASOURCES.values["_one_nested_yaml"] = MARKET_DATASOURCES.Value("YAML with one nesting")

    class literal_unicode(unicode):
        pass

    def literal_unicode_representer(self, dumper, data):
        return dumper.represent_scalar(u'tag:yaml.org,2002:str', data, style='|')

    def _java_properties(self, dirname, filename, environment):
        z = dict()
        lines = [line.strip() for line in open(os.path.join(dirname, environment, "etc", filename))]
        for line in (line for line in lines if not line.startswith(("#", "!")) and line.strip()):
            kv = map(str.strip, line.split("=", 1))
            z.update({kv[0]: kv[1]})
        return z

    def _cfg(self, dirname, filename, environment):
        z = dict()
        Config = ConfigParser.ConfigParser()
        Config.read(os.path.join(dirname, environment, "etc", filename))
        for section in Config.sections():
            for option in Config.options(section):
                key = section + "/" + option
                z.update({key: Config.get(section, option)})
        return z

    def _datasources_xml(self, dirname, filename, environment):
        z = dict()
        xml = ET.parse(os.path.join(dirname, environment, "etc", filename))
        xmlroot = xml.getroot()
        for section in xmlroot[0]:
            for key in section:
                value = key.text
                key = section.tag + "/" + key.tag
                z.update({key: value})
        return z

    def _mbi_endpoints_ent(self, dirname, filename, environment):
        z = dict()
        lines = [line.strip() for line in open(os.path.join(dirname, environment, "etc", filename))]
        for line in (line.replace(">", "").replace("\"", "") for line in lines if
                     line.startswith(("<!ENTITY ")) and line.strip()):
            kv = map(str.strip, line.split())
            z.update({kv[1]: kv[2]})
        return z

    def _one_nested_yaml(self, dirname, filename, environment):
        z = dict()
        ds = yaml.safe_load(open(os.path.join(dirname, environment, "etc", filename)))
        for section in ds:
            for option in ds[section]:
                key = section + "/" + option
                value = ds[section][option]
                z.update({key: value})
        return z

    def _plain_text(self, dirname, filename, environment):
        z = dict()
        with open(os.path.join(dirname, environment, "etc", filename)) as f:
            key = "content"
            value = f.read()
            z.update({key: self.literal_unicode(value)})
        return z

    def _git_clone(self, repo):
        logging.info("Cloning repo {}".format(repo))
        with Key(self, VAULT_OWNER, VAULT_SSH):
            dirname = mkdtemp()
            git = Git(repo)
            git.clone(dirname, "master")
            return dirname

    def _git_push(self, dirname, environment):
        with Key(self, VAULT_OWNER, VAULT_SSH):
            with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("git_clone")) as pl:
                pr1 = sp.Popen("git diff --quiet", shell=True, cwd=dirname, stdout=pl.stdout, stderr=sp.STDOUT).wait()
                if pr1 == 0:
                    logging.info("There is no changes")
                    return
                pr2 = sp.Popen("git add {}".format(os.path.join(environment, "etcd.yml")), shell=True, cwd=dirname, stdout=pl.stdout, stderr=sp.STDOUT).wait()
                if pr2 != 0:
                    raise common.errors.TaskError("Can't add etcd.yml")
                pr3 = sp.Popen("git commit -m 'Sync via sandbox task' --author='csadmin <csadmin@yandex-team.ru>'", shell=True, cwd=dirname, stdout=pl.stdout, stderr=sp.STDOUT).wait()
                if pr3 != 0:
                    raise common.errors.TaskError("Can't commit changes")
                pr4 = sp.Popen("git push", shell=True, cwd=dirname, stdout=pl.stdout, stderr=sp.STDOUT).wait()
                if pr4 != 0:
                    raise common.errors.TaskError("Can't push changes")

    def on_execute(self):
        e = self.Parameters.environment
        environment = e.decode("utf8").encode("ascii")
        MARKET_DATASOURCES = self.Parameters.MARKET_DATASOURCES

        datasources_dir = self._git_clone(DATASOURCES)
        datasources_ng_dir = self._git_clone(DATASOURCES_NG)

        ks = dict()
        for k, v in MARKET_DATASOURCES.items():
            k = k.decode("utf8").encode("ascii")
            ks[k] = getattr(self, v)(datasources_dir, k, environment)

        ks = dict({"datasources": {environment: ks}})
        yaml.add_representer(self.literal_unicode, self.literal_unicode_representer)
        with open(os.path.join(datasources_ng_dir, environment, "etcd.yml"), "w") as f:
            yaml.dump(ks, f, default_flow_style=False, allow_unicode=True, encoding='utf-8')

        self._git_push(datasources_ng_dir, environment)
