# -*- coding: utf-8 -*-
"""This task copy gencfg groups"""

import os
import re
import logging
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import subprocess as sp


@contextmanager
def _chdir(path):
    curdir = os.path.abspath(os.curdir)
    os.chdir(path)
    yield
    os.chdir(curdir)


class MarketCopyGencfgGroup(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        arcadia_user = "zomb-sandbox-rw"
        args = sdk2.parameters.String('Arguments to run the script', required=True)
        startrek_ticket = sdk2.parameters.String("Startrek ticket", required=False)
        with sdk2.parameters.Output:
            commit_url = sdk2.parameters.String("Commit URL", default="", required=True)

    def on_execute(self):
        self._prepare_env()
        dest_group = self._get_dest_group()
        if self._is_group_exist(dest_group):
            logging.info('Group %s already exists', dest_group)
            self.Parameters.commit_url = "Not changed"
            return
        self._create_groups()
        self._run_checks()
        self._commit_changes()

    def _prepare_env(self):
        logging.info('Preparing an environment for GenCfg')

        gencfg_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, 'gencfg')
        self.gencfg_path = Arcadia.checkout(gencfg_url, str(self.path('gencfg')))
        logging.info('Arcadia is checkouted to %s', self.gencfg_path)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('install')) as pl:
            process = sp.Popen(
                ['bash', './install.sh'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('update')) as pl:
            process = sp.Popen(
                ['bash', './update.sh'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

    def _get_dest_group(self):
        args = self.Parameters.args.split(' ')
        for i, arg in enumerate(args):
            if arg == '-d':
                return args[i + 1]

    def _is_group_exist(self, dest_group):
        logging.info('Check is group {} exists'.format(dest_group))
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('is_group_exist')) as pl:
            process = sp.Popen(
                ' '.join(['/skynet/python/bin/python ./utils/common/show_group_card.py -g', dest_group]),
                shell=True,
                cwd=self.gencfg_path,
                stdout=sp.PIPE,
                stderr=pl.stdout
            )
            process.wait()
        outlines = process.communicate()[0].split('\n')
        if outlines[0].endswith('does not exist'):
            return False
        return True

    def _create_groups(self):
        logging.info('Creating groups')

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('copy_groups')) as pl:
            process = sp.Popen(
                ' '.join(['/skynet/python/bin/python', './utils/market/copy_group.py', self.Parameters.args]),
                shell=True,
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()

        if process.returncode:
            raise TaskFailure('Failed to create new group')

    def _run_checks(self):
        logging.info('Running checks')
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('run_checks')) as pl:
            process = sp.Popen(
                ['bash', './gen.sh', 'run_checks'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

    def _commit_changes(self):
        logging.info('Commiting changes')

        groups_path = 'db'

        if self.Parameters.startrek_ticket:
            ticket = self.Parameters.startrek_ticket + " "
        else:
            ticket = ""

        with _chdir(os.path.join(self.gencfg_path, groups_path)):
            logging.info('Svn status: %s', Arcadia.status(path='.'))
            logging.info('Svn diff: \n%s', Arcadia.diff('.'))

            output = Arcadia.commit(path='.',
                                    message='{}Create new group'.format(ticket),
                                    user=self.Parameters.arcadia_user)
            if not output:
                raise TaskFailure('Nothing has been commited')
            revision = re.findall(r'Committed revision (\d+)\.', output)
            if revision:
                self.Parameters.commit_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])

        logging.info('Changes have been commited')
