# -*- coding: utf-8 -*-
import os
import yaml
import logging

import sandbox.common.types.task as ctt
from sandbox.sdk2 import Task
from sandbox.sdk2 import parameters
from sandbox.sdk2.vcs import svn as sdk2_svn
from sandbox.sdk2 import WaitTask
from sandbox.projects.market.sre.RunMarketDataGetter import RunMarketDataGetter


class MarketDataGetterManager(Task):

    class Parameters(Task.Parameters):
        skip_stable = parameters.Bool("Skip configs for stable", default=False)
        skip_prestable = parameters.Bool("Skip configs for prestable", default=False)
        skip_testing = parameters.Bool("Skip configs for testing", default=False)
        config_dir_arcadia_url = parameters.ArcadiaUrl(
            "Path to config dir in Arcadia",
            default_value="arcadia:/arc/trunk/arcadia/market/sre/conf/sandbox-data-getter",
            required=True)
        yt_token_vault = parameters.Vault("Vault secret contains YT token", required=False)
        arc_token_vault = parameters.Vault("Vault secret contains ARC_TOKEN", required=False)


    def parse_configs(self):
        checkout_path = sdk2_svn.Arcadia.get_arcadia_src_dir(self.Parameters.config_dir_arcadia_url)
        configs = {}
        for config_name in [f for f in os.listdir(checkout_path) if os.path.isfile(os.path.join(checkout_path, f))]:
            logging.debug("Parsing config {}".format(config_name))
            if not config_name.endswith(".yaml"):
                logging.info("Skipping {} config file: must ends with '.yaml'".format(config_name))
                continue
            with open(os.path.join(checkout_path, config_name), "r") as fp:
                try:
                    configs[config_name] = yaml.load(fp)
                except Exception as e:
                    logging.error("Error parsing {} config file".format(config_name), e)
        return configs

    def on_execute(self):
        with self.memoize_stage.first_step:
            subtasks = self.run_data_getter_tasks()
            if not subtasks:
                return
            raise WaitTask(subtasks, ctt.Status.Group.FINISH | ctt.Status.Group.BREAK, wait_all=True)

    def run_data_getter_tasks(self):
        configs = self.parse_configs()

        subtasks = []
        for config_name, config in configs.iteritems():
            is_enabled = config["enable"]
            if not is_enabled:
                continue

            bundle_name = config["bundle_name"]
            env_list = config["env_list"]
            resource_type = config["resource_type"]
            resources = config["resources"]

            reqs = config.get("reqs", {})
            disk_reqs = int(reqs.get("disk", 3))

            advanced_settings = config.get("advanced_settings", {})
            is_stable_disabled = advanced_settings.get("disable_stable", False)
            is_prestable_disabled = advanced_settings.get("disable_prestable", False)
            is_testing_disabled = advanced_settings.get("disable_testing", False)
            save_resource_cache = advanced_settings.get("save_resource_cache", False)
            use_resource_cache = advanced_settings.get("use_resource_cache", False)
            create_stat_file = advanced_settings.get("create_stat_file", False)
            create_meta_file = advanced_settings.get("create_meta_file", False)
            create_nanny_release = advanced_settings.get("create_nanny_release", True)
            create_deploy_release = advanced_settings.get("create_deploy_release", False)
            archive_bundle = advanced_settings.get("archive_bundle", False)

            for env_name in env_list:
                if env_name == "stable" and (is_stable_disabled or self.Parameters.skip_stable):
                    continue
                if env_name == "prestable" and (is_prestable_disabled or self.Parameters.skip_prestable):
                    continue
                if env_name == "testing" and (is_testing_disabled or self.Parameters.skip_testing):
                    continue

                child = RunMarketDataGetter(
                    self,
                    description="Формирование data-getter бандла {} в {}".format(bundle_name, env_name),
                    disk_reqs=disk_reqs,
                    target_service=bundle_name,
                    resource_type_name=resource_type,
                    resources="\n".join(resources),
                    environment_type=env_name,
                    use_getter_latest_version=True,
                    use_xmllint_latest_version=True,
                    create_nanny_release=create_nanny_release,
                    create_deploy_release=create_deploy_release,
                    create_resource_bundle=True,
                    archive_bundle=archive_bundle,
                    create_stat_file=create_stat_file,
                    create_meta_file=create_meta_file,
                    save_cache=save_resource_cache,
                    use_resource_cache=use_resource_cache,
                    yt_token_vault=self.Parameters.yt_token_vault,
                    arc_token_vault=self.Parameters.arc_token_vault,
                )
                child.Requirements.disk_space = disk_reqs * 1024
                child.save()
                child.enqueue()
                subtasks.append(child)

        return subtasks
