"""
-*- coding: utf-8 -*-
for creating pbuilder image:
1) pbuilder --crease trusty.tgz --distribution trusty
2) sudo pbuilder --login --save-after-login --basetgz trusty.tgz
3) rm /dev/loop2 /dev/port /dev/loop7 /dev/midi02 /dev/tty4 /dev/loop6 /dev/tty8 /dev/loop4 /dev/ram10 /dev/dsp3
/dev/ram12 /dev/ram8 /dev/tty9 /dev/ram7 /dev/mpu401data /dev/rmidi2 /dev/ram2 /dev/loop0 /dev/loop1 /dev/midi2
/dev/ram9 /dev/tty3 /dev/dsp /dev/midi0 /dev/ram15 /dev/midi1 /dev/smpte3 /dev/tty2 /dev/kmem /dev/rmidi0 /dev/ram3
/dev/mixer /dev/ram6 /dev/loop5 /dev/audio1 /dev/dsp2 /dev/agpgart /dev/full /dev/tty6 /dev/mem /dev/mixer3
/dev/mpu401stat /dev/audio3 /dev/midi3 /dev/dsp1 /dev/sequencer /dev/midi01 /dev/audio2 /dev/midi00 /dev/audio
/dev/ram14 /dev/rmidi3 /dev/sndstat /dev/smpte2 /dev/ram1 /dev/tty5 /dev/loop3 /dev/midi03 /dev/smpte0 /dev/mixer1
/dev/ram4 /dev/mixer2 /dev/ram0 /dev/ram11 /dev/ram13 /dev/audioctl /dev/ram5 /dev/ram16 /dev/smpte1 /dev/tty7
/dev/rmidi1  (SANDBOX-2958)
4) ya upload trusty.tgz -T MARKET_PBUILDER_IMAGE --do-not-remove -A release=trusty
"""
import logging
import os
import re
import time

import yaml
from sandbox.common import errors
from sandbox.common.types.client import Tag
from sandbox.projects import resource_types
from sandbox.projects.common import utils
from sandbox.projects.common.arcadia import sdk as arcadiasdk
from sandbox.projects.common.debpkg import DebRelease
from sandbox.projects.common.gnupg import GpgKey
from sandbox.projects.market.resources import MARKET_PBUILDER_IMAGE
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.errors import SandboxTaskFailureError
from sandbox.sandboxsdk.parameters import SandboxStringParameter, SandboxBoolParameter, SandboxGitUrlParameter, \
    SandboxArcadiaUrlParameter
from sandbox.sandboxsdk.process import run_process, SandboxSubprocessError
from sandbox.sandboxsdk.ssh import Key
from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sdk2.vcs.git import Git

import conductor
import resources
from dist import Dist

SUPPORTED_UBUNTU_VERSIONS = ['precise', 'trusty', 'xenial']
DEFAULT_UBUNTU_VERSION = 'trusty'
DEFAULT_DISTRIBUTION = 'unstable'
DEFAULT_TESTING_DISTRIBUTION = 'testing'
WAIT_REPO_TIMEOUT = 600

REPOS = [
    'common',
    'market',
    'market-common',
    'market-precise',
    'market-trusty',
    'delivery',
    'hadoop',
    'load',
    'load-trusty',
    'load-precise',
    'load-xenial',
]


class ArcadiaURL(SandboxArcadiaUrlParameter):
    name = 'ArcadiaURL'
    description = 'URL to repository with source files. Type `svn info` and copy-paste URL'
    required = True
    group = 'Source parameters'


class GitURL(SandboxGitUrlParameter):
    name = 'GitURL'
    description = 'URL to repository with source files. Type `git remote -v` and copy-paste URL'
    required = True
    group = 'Source parameters'


class GitBranch(SandboxStringParameter):
    name = 'GitBranch'
    description = 'Git branch:'
    default_value = 'master'
    required = True
    group = 'Source parameters'


class RepoType(SandboxStringParameter):
    name = 'RepoType'
    description = 'Type of repo'
    required = True
    choices = [('arcadia', 'arcadia.yandex.ru'), ('github', 'github.yandex-team.ru')]
    default_value = 'github.yandex-team.ru'
    sub_fields = {
        'arcadia.yandex.ru': [ArcadiaURL.name],
        'github.yandex-team.ru': [GitURL.name, GitBranch.name]
    }
    group = 'Source parameters'


class BuildParamPath(SandboxStringParameter):
    name = "BuildParamPath"
    description = "Path to directory with .build.yml"
    default_value = ""
    group = 'Source parameters'


class CheckoutSSHKey(SandboxStringParameter):
    name = 'CheckoutSSHKey'
    description = 'Name of SSH key in Vault for checkout sources'
    default_value = 'csadmin'
    required = True
    group = 'Source parameters'


class KeyOwner(SandboxStringParameter):
    name = 'KeyOwner'
    description = 'Owner name of SSH key in Vault for checkout sources'
    default_value = 'MARKETSRE'
    required = True
    group = 'Source parameters'


class BuildDistr(SandboxStringParameter):
    name = 'BuildDistr'
    description = 'Name of image for pdebuild'
    group = 'Build parameters'
    default_value = 'trusty'


class BuildHooks(SandboxStringParameter):
    name = 'BuildHooks'
    description = 'Dir name for pdebuild hooks'
    group = 'Build parameters'
    default_value = 'hooks.d'


class BuildDupload(SandboxBoolParameter):
    name = 'BuildDupload'
    description = 'Dupload packages'
    group = 'Dist parameters'


class BuildRepoName(SandboxStringParameter):
    name = 'BuildRepoName'
    description = 'Repo name'
    choices = [(repo, repo) for repo in REPOS]
    group = 'Dist parameters'
    default_value = 'market-common'


class BuildRepoCheckIfExist(SandboxBoolParameter):
    name = 'BuildRepoCheckIfExist'
    description = 'Check if exist in repo'
    group = 'Dist parameters'
    default_value = 'True'


class BuildRepoUser(SandboxStringParameter):
    name = 'BuildRepoUser'
    description = 'Name of user for sign and dupload'
    group = 'Dist parameters'
    default_value = 'robot-market-dist'


class BuildConductorCreateTicket(SandboxBoolParameter):
    name = 'BuildConductorCreateTicket'
    description = 'Create ticket in conductor'
    group = 'Conductor parameters'
    default_value = False


class BuildConductorBranch(SandboxStringParameter):
    name = 'BuildConductorBranch'
    description = 'Branch in conductor'
    choices = [(branch, branch) for branch in ['unstable', 'testing', 'prestable', 'stable', 'hotfix', 'fallback']]
    group = 'Conductor parameters'
    default_value = 'unstable'


class BuildConductorProjects(SandboxStringParameter):
    name = 'BuildConductorProjects'
    description = 'Projects for tickets in conductor'
    group = 'Conductor parameters'


class BuildParamOverride(SandboxBoolParameter):
    name = 'BuildParamOverride'
    description = 'Override build params'
    default_value = False
    group = 'Build parameters'
    sub_fields = {
        'true': [BuildDistr.name, BuildHooks.name, BuildDupload.name, BuildRepoName.name, BuildRepoCheckIfExist.name,
                 BuildRepoUser.name, BuildConductorCreateTicket.name, BuildConductorBranch.name,
                 BuildConductorProjects.name]
    }


class MarketDebuilder(SandboxTask):
    privileged = True
    type = 'MARKET_DEBUILDER'
    client_tags = Tag.LINUX_PRECISE | Tag.LINUX_TRUSTY
    input_parameters = (
        RepoType,
        GitURL,
        GitBranch,
        ArcadiaURL,
        BuildParamPath,
        CheckoutSSHKey,
        KeyOwner,
        BuildParamOverride,
        BuildDistr,
        BuildHooks,
        BuildDupload,
        BuildRepoName,
        BuildRepoCheckIfExist,
        BuildRepoUser,
        BuildConductorCreateTicket,
        BuildConductorBranch,
        BuildConductorProjects,
    )
    workdir = 'workdir'
    build_options = {}

    def on_prepare(self):
        with Key(self, utils.get_or_default(self.ctx, KeyOwner), '{}-ssh'.format(utils.get_or_default(self.ctx, CheckoutSSHKey))):
            while not self.ctx.get('source_cloned', False):
                self._get_sources()

    def on_execute(self):
        self.build_options = self._get_build_options()
        self._prepare_environ()
        self._build()

    def _get_basetgz(self, ubuntu_version='trusty'):
        pbimage = resources.get_newest_resource(self, MARKET_PBUILDER_IMAGE, ubuntu_version,
                                                attrs={'release': ubuntu_version})
        logging.info('pbimage with release {} synced to {}'.format(ubuntu_version, pbimage))
        return pbimage

    def _get_sources(self):
        logging.info('Selected repo {}'.format(self.ctx.get(RepoType.name)))
        return self.repo_type_map.get(self.ctx.get(RepoType.name))(self)

    def _clone_from_git(self):
        if not self.ctx.get('source_cloned'):
            url = self.ctx.get(GitURL.name, '')
            branch = self.ctx.get(GitBranch.name, 'master')
            logging.info('Clone git repo from {} {}'.format(url, branch))
            git = Git(url)
            git.clone(self.workdir, branch)
            self.ctx['source_cloned'] = True
            return True

    def _clone_from_arcadia(self):
        if not self.ctx.get('source_cloned'):
            logging.info('Clone arcadia repo from {}'.format(self.ctx.get(ArcadiaURL.name)))
            src_dir_root = arcadiasdk.do_clone(self.ctx.get(ArcadiaURL.name), self)
            self.ctx['source_cloned'] = True
            self.workdir = src_dir_root
            return True

    def _get_build_options(self):
        return BuildOptions(os.path.join(self.workdir.rstrip('/'),
                                         utils.get_or_default(self.ctx, BuildParamPath).strip('/'),
                                         '.build.yml'),
                            self.ctx)

    def _prepare_environ(self):
        for v in ['TEMP', 'TMPDIR', 'TMP']:
            if v in os.environ:
                del os.environ[v]

        # it used for debsign
        repo_user = self.build_options.params.get('repo', {}).get('user')
        os.environ['DEBFULLNAME'] = repo_user
        os.environ['EMAIL'] = '{}@yandex-team.ru'.format(repo_user)

        # it used for arcadia
        os.environ['SVN_SSH'] = 'ssh -o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null'

        logging.debug(os.environ)
        install_pbuilder_cmd = "sudo apt-get update && sudo apt-get install " \
                               "pbuilder -q0 --assume-yes --force-yes -o Dpkg::Options::='--force-confdef'"
        run_process(
            install_pbuilder_cmd,
            shell=True, log_prefix='install_pbuilder')

    def _build(self):
        debian_abs_path = os.path.join(self.workdir, self.ctx.get(BuildParamPath.name).lstrip('/'), 'debian')
        source_path = os.path.join(self.workdir, self.ctx.get(BuildParamPath.name).lstrip('/'))
        with open(os.path.join(debian_abs_path, 'control'), 'r') as f:
            control_text = f.read()
            source_package = re.search('^Source:\s*(.*)\n', control_text).group(1)
            binary_packages = re.findall('^Package:[\t ]*(.+)$', control_text, re.MULTILINE)

        with open(os.path.join(debian_abs_path, 'changelog'), 'r') as f:
            version, distribution = re.search('^[\.\w_-]+ \((.+?)\) (\w+)', f.read()).groups()
        self.ctx['PackageVersion'] = version
        ubuntu_version = self.build_options.params.get('distr')
        # make cmds
        build_cmd = " ".join([
            'cd ' + source_path + ' && ',
            'pdebuild',
            '--use-pdebuild-internal',
            '--auto-debsign --debsign-k {}'.format(self.build_options.params.get('repo', {}).get('user')),
            '--buildresult ' + self.abs_path(),
            '--',
            '--basetgz ' + self._get_basetgz(ubuntu_version),
            '--hookdir ' + self.build_options.params.get('hookdir', '{}/hooks.d/'.format(self.workdir)),
            '--buildplace ' + self.abs_path() + ' --aptcache ' + self.abs_path(),
        ])
        version_wo_epoch = re.search('^\d+:(.*)', version)
        version_wo_epoch = version_wo_epoch.group(1) if version_wo_epoch else version
        repo = self.build_options.params.get('repo', {}).get('name')
        upload_cmd = "dupload --force --to {repo} {abs_path}/{source_package}_{version_wo_epoch}_a*.changes".format(
            repo=repo,
            abs_path=self.abs_path(),
            source_package=source_package,
            version_wo_epoch=version_wo_epoch
        )
        check_if_exist = self.build_options.params.get('repo', {}).get('check_if_exist')
        dist = Dist()
        repo_user = self.build_options.params.get('repo', {}).get('user')
        DUPLOAD_CONF = {
            r: {
                'fqdn': 'dupload.dist.yandex.ru',
                'method': 'scpb',
                'incoming': '/repo/{}/mini-dinstall/incoming/'.format(r),
                'dinstall_runs': 0,
                'login': repo_user,
                'options': '-o StrictHostKeyChecking=no -o UserKnownHostsFile=/dev/null',
            } for r in REPOS
        }
        if not check_if_exist or not dist.if_exist(binary_packages[0], version, repo):
            with GpgKey(self, utils.get_or_default(self.ctx, KeyOwner), "{}-gpg-private".format(repo_user), "{}-gpg-public".format(repo_user)):
                logging.info('Build with cmd:\n{}'.format(build_cmd))
                process_info = run_process(build_cmd, shell=True, check=False, log_prefix='pdebuild')
                if process_info.returncode > 0:
                    stdout = open(process_info.stdout_path, 'r').read()
                    if 'Some index files failed to download' in stdout:
                        raise errors.TemporaryError('Temporary problems with repos')
                    else:
                        raise SandboxSubprocessError(
                            message='Build package failed',
                            stdout_path=process_info.stdout_path_filename,
                            logs_resource=getattr(getattr(channel.task, '_log_resource', None), 'id', None),
                        )
                if self.build_options.params.get('repo', {}).get('dupload'):
                    with DebRelease(DUPLOAD_CONF):
                        with Key(self, utils.get_or_default(self.ctx, KeyOwner), '{}-ssh'.format(repo_user)):
                            run_process('set -x; chmod 0644 *.{dsc,deb,tar.gz,changes}', shell=True, log_prefix='chmod')
                            run_process(upload_cmd, shell=True, log_prefix='dupload')

                    start_time = time.time()
                    packages_were_found = False
                    while not packages_were_found and time.time() < start_time + WAIT_REPO_TIMEOUT:
                        time.sleep(10)
                        logging.debug("Checking packages in repo ...")
                        packages_were_found = dist.if_exist(binary_packages[0], version, repo)

                    if not packages_were_found:
                        raise errors.TaskError(
                            "Packages are not found in {} after {} secs from dupload".format(repo, WAIT_REPO_TIMEOUT))

                    for binary_package in binary_packages:
                        self.set_info(
                            'Package {}={} successfully uploaded to {}/{}'.format(binary_package, version, repo,
                                                                                  distribution))
                    if self.build_options.params.get('conductor', {}).get('create_ticket'):
                        cond_branch = self.build_options.params.get('conductor', {}).get('branch')
                        conductor.create_conductor_ticket(self, binary_packages, version, branch=cond_branch)
                for binary_package in binary_packages:
                    self.set_info('{} builded'.format(binary_package))
                    for cpu_type in ['all', 'amd64']:
                        resource_path = '{abs_path}/{binary_package}_{version_wo_epoch}_{cpu_type}.deb)'.format(
                            abs_path=self.abs_path(),
                            binary_package=binary_package,
                            version_wo_epoch=version_wo_epoch,
                            cpu_type=cpu_type)
                        if os.path.exists(os.path.basename(resource_path)):
                            self.create_resource(description='Package ' + binary_package + '=' + version,
                                                 resource_path=resource_path,
                                                 resource_type=resource_types.DEB_PACKAGE_TAR,
                                                 attributes={'package_name': binary_package, 'version': version},
                                                 owner=self.owner)
                        else:
                            logging.info('{} not found in {}'.format(os.path.basename(resource_path), os.getcwd()))
        else:
            for binary_package in binary_packages:
                self.set_info('Package {}={} is already in {}'.format(binary_package, version, repo))

    repo_type_map = {
        'arcadia.yandex.ru': _clone_from_arcadia,
        'github.yandex-team.ru': _clone_from_git,
    }


class BuildOptions(object):
    """
    Class for construct and validate build options
    """
    build_options = {
        'distr': 'trusty',
        'hookdir': 'hooks.d',
        'repo': {
            'name': 'common',
            'check_if_exist': False,
            'dupload': False,
            'user': 'robot-market-dist'
        },
        'conductor': {
            'create_ticket': False,
            'branch': 'unstable',
            'projects': [],
        }
    }

    def __init__(self, build_file, ctx):
        logging.info('Trying to load .build.yml')
        self.ctx = ctx
        self.build_options = self._read_file_(build_file)
        self.params = self._merge_()
        logging.info(self.params)

    def _read_file_(self, build_file):
        if self.ctx.get(BuildParamOverride.name):
            logging.info('Params overrided')
            return self.build_options
        try:
            with open(build_file, 'r') as stream:
                logging.info('Try to read .build.yml')
                return yaml.load(stream)
        except IOError:
            logging.info('.build.yml not found')
            raise SandboxTaskFailureError('build.yml not found, and not overrided in params')

    def _merge_(self):
        repo_params = self.build_options.get('repo')
        conductor_params = self.build_options.get('conductor')
        build_params = {
            'distr': self.ctx.get(BuildDistr.name, self.build_options.get('distr')),
            'hookdir': self.build_options.get('hookdir'),
            'repo': {
                'name': self.ctx.get(BuildRepoName.name, repo_params.get('name')),
                'check_if_exist': self.ctx.get(BuildRepoCheckIfExist.name, repo_params.get('check_if_exist')),
                'dupload': self.ctx.get(BuildDupload.name, repo_params.get('dupload')),
                'user': self.ctx.get(BuildRepoUser.name, repo_params.get('user')),
            },
            'conductor': {
                'create_ticket': self.ctx.get(BuildDupload.name, conductor_params.get('create_ticket')),
                'branch': self.ctx.get(BuildConductorBranch.name, conductor_params.get('branch')),
                'projects': self.ctx.get(BuildConductorProjects.name, conductor_params.get('projects'))
            }
        }
        return build_params


__Task__ = MarketDebuilder
