# -*- coding: utf-8 -*-
from os import path, makedirs

from sandbox import sdk2
from sandbox.common.types import client as ctc
from sandbox.projects.common.task_env import TinyRequirements
from sandbox.projects.common.vcs.arc import Arc

ARCANUM_REVIEW_BASE = "https://a.yandex-team.ru/review/"


class Status(object):
    NOOP = "NOOP"
    FAILED = "FAILED"
    COMMITTED = "COMMITTED"


class MarketDeplateGenerateServiceFolders(sdk2.Task):
    """
        Generate Service with Deplate
    """

    class Requirements(TinyRequirements):
        disk_space = 1 * 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Deplate Generate", collapse=False) as deplate_generate_block:
            arc_token = sdk2.parameters.YavSecret(
                "Arc token",
                required=True
            )
            arc_user = sdk2.parameters.String('Arc username', required=True)
            ticket = sdk2.parameters.StrictString("Startrek ticket", required=True, regexp=r'[A-Z]+-[0-9]+')
            root_abc = sdk2.parameters.String("Root abc (child of market_map)", required=True)

            with sdk2.parameters.Output:
                result_status = sdk2.parameters.String("Result")
                result_pr = sdk2.parameters.Integer("Arcadia PR")

    def _arc(self):
        key = 'token'
        if self.Parameters.arc_token.default_key:
            key = self.Parameters.arc_token.default_key

        return Arc(arc_oauth_token=self.Parameters.arc_token.data()[key])

    @staticmethod
    def _get_pr_num(out):
        if ARCANUM_REVIEW_BASE not in out:
            raise ValueError("invalid output from pr create: {}".format(out))
        pr_num = out.split(ARCANUM_REVIEW_BASE)[1].strip()
        return int(pr_num)

    def on_execute(self):
        arc = self._arc()
        ticket = str(self.Parameters.ticket)
        service_dir = path.join("market/sre/conf/deplate/values", str(self.Parameters.root_abc).lower())

        with arc.mount_path("", "trunk", fetch_all=False) as mp:
            arc.checkout(mp, branch=ticket, create_branch=True)

            service_path = path.join(mp, service_dir)
            created = False
            if not path.exists(service_path):
                created = True
                makedirs(service_path)
                arc.add(mp, service_path)

            for env in {"unstable", "testing", "prestable", "production"}:
                env_path = path.join(service_path, env)
                if not path.exists(env_path):
                    created = True
                    makedirs(env_path)
                    with open(path.join(env_path, '.deplate'), 'w'):
                        pass
                    arc.add(mp, env_path)

            if not created:
                self.Parameters.result_pr = 0
                self.Parameters.result_status = Status.NOOP
                return

            commit_message = "{}: generate folders for {}".format(ticket, self.Parameters.root_abc)
            arc.commit(mp, commit_message)

            arc.push(mp, upstream="users/{}/{}-deplate".format(self.Parameters.arc_user, ticket), force=True)

            pr_num = self._get_pr_num(
                arc.pr_create(mp, publish=False, auto=True, message=commit_message, no_commits=True))
            self.Parameters.result_pr = pr_num
            self.Parameters.result_status = Status.COMMITTED
            self.set_info(
                'Arcanum <a href="{}{}">PR:{}</a> for Deplate service config folders.'.format(ARCANUM_REVIEW_BASE,
                                                                                              pr_num, pr_num),
                do_escape=False)
