# -*- coding: utf-8 -*-

import logging
from os import path

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types import client as ctc
from sandbox.projects.common.task_env import TinyRequirements
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.market.resources.sre import MarketDeplateBinary
from sandbox.sdk2.helpers import subprocess as sp


class MarketDeplateRun(sdk2.Task):
    """
        Run Deplate
    """

    class Requirements(TinyRequirements):
        disk_space = 1 * 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("Deplate", collapse=False) as deplate_block:
            deplate_binary = sdk2.parameters.Resource(
                'Deplate binary',
                resource_type=MarketDeplateBinary,
                required=True,
            )
            deplate_command = sdk2.parameters.String('Deplate command')
            deplate_token = sdk2.parameters.YavSecret(
                "Deplate token",
                required=True
            )
            arc_token = sdk2.parameters.YavSecret(
                "Arc token",
                required=True
            )
            arc_user = sdk2.parameters.String('Arc username', required=True)

    def on_create(self):
        resource = MarketDeplateBinary.find(attrs={"released": "stable"}).first()
        if resource:
            self.Parameters.deplate_binary = resource

    def _prepare_deplate(self):
        if not self.Context.deplate_path or not self.Context.deplate_token:
            data = sdk2.ResourceData(self.Parameters.deplate_binary)
            self.Context.deplate_path = path.join(str(data.path), "deplate")

            key = 'token'
            if self.Parameters.deplate_token.default_key:
                key = self.Parameters.deplate_token.default_key

            self.Context.deplate_token = self.Parameters.deplate_token.data()[key]

    def _run_command(self, args, arc_path, valid_codes=None, assume_yes=False):
        self._prepare_deplate()

        env = {
            "DEPLATE_TOKEN": self.Context.deplate_token,
            "DEPLATE_ARC_PATH": arc_path,
        }
        if assume_yes:
            env["DEPLATE_ALWAYS_ASSUME_YES"] = "true"

        if valid_codes:
            valid_codes = set(valid_codes)
            valid_codes.add(0)
        else:
            valid_codes = {0}

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("deplate")) as pl:
            pr = sp.Popen([self.Context.deplate_path] + args, stdout=pl.stdout,
                          stderr=pl.stderr, env=env).wait()

            if pr not in valid_codes:
                raise TaskError("Failed to run deplate")
            return pr

    def _arc(self):
        key = 'token'
        if self.Parameters.arc_token.default_key:
            key = self.Parameters.arc_token.default_key

        return Arc(arc_oauth_token=self.Parameters.arc_token.data()[key])

    def on_execute(self):
        arc = self._arc()

        with arc.mount_path("", "trunk", fetch_all=False) as mp:
            self._run_command(str(self.Parameters.deplate_command).split(), mp)
