# -*- coding: utf-8 -*-
from os import path, makedirs

from sandbox import sdk2

from sandbox.projects.market.sre.MarketDeplateRun import MarketDeplateRun

ARCANUM_REVIEW_BASE = "https://a.yandex-team.ru/review/"


class Status(object):
    NOOP = "NOOP"
    FAILED = "FAILED"
    COMMITTED = "COMMITTED"


class MarketDeplateRunGenerate(MarketDeplateRun):
    """
        Generate Service with Deplate
    """

    class Parameters(MarketDeplateRun.Parameters):
        deplate_command = None

        with sdk2.parameters.Group("Deplate Generate", collapse=False) as deplate_generate_block:
            ticket = sdk2.parameters.StrictString("Startrek ticket", required=True, regexp=r'[A-Z]+-[0-9]+')

            info = sdk2.parameters.Info("info")
            service_name = sdk2.parameters.String("Service name", required=True)
            app_name = sdk2.parameters.String("App name", required=True)
            root_abc = sdk2.parameters.String("Root abc (child of market_map)", required=True)
            abc = sdk2.parameters.String("ABC ID or slug", required=True)
            with sdk2.parameters.RadioGroup("Environment", required=True) as environment:
                environment.values.unstable = None
                environment.values.testing = None
                environment.values.prestable = None
                environment.values.production = None

            resources = sdk2.parameters.Info("resources")
            cpu = sdk2.parameters.Integer("VCPU limit, 1 core equal 1000 VCPU. 100 is minimum", required=True)
            ram = sdk2.parameters.Integer("Ram limit in Gb. 1gb is minimum", required=True)
            with sdk2.parameters.RadioGroup("Disk type", required=True) as disk_type:
                disk_type.values.ssd = None
                disk_type.values.hdd = disk_type.Value(default=True)
            disk_size = sdk2.parameters.Integer("Disk limit in Gb. 10gb is minimum", required=True)

            pods_count = sdk2.parameters.Info("pods count")
            sas_pods = sdk2.parameters.Integer("SAS pods", required=True, default=0)
            vla_pods = sdk2.parameters.Integer("VLA pods", required=True, default=0)
            man_pods = sdk2.parameters.Integer("MAN pods", required=True, default=0)

            configuration = sdk2.parameters.Info("configuration")
            app_command = sdk2.parameters.String("Command to run app")
            network_macro = sdk2.parameters.StrictString(
                "Network macros where to start service, must start and end with '_'", required=True, regexp=r'_[^_]+_')

            templates = sdk2.parameters.Info("templates")
            template_root = sdk2.parameters.String("Root template", default="base.jsonnet")
            template_modifiers = sdk2.parameters.List("List of template modifiers",
                                                      default=["box/infra.jsonnet", "box/base_app.jsonnet",
                                                               "remove_duplicates.jsonnet", "update_layers.jsonnet"])
            template_tools = sdk2.parameters.List("List of template tools", default=["tools.libjsonnet"])
            exp3_consumer = sdk2.parameters.String("Consumer for exp3", default="")
            exp3_secret_id = sdk2.parameters.String("secret id for exp3 matcher", default="")
            exp3_secret_version = sdk2.parameters.String("secret version for exp3 matcher", default="")
            exp3_secret_alias = sdk2.parameters.String("secret alias for exp3 matcher", default="")
            with sdk2.parameters.Output:
                result_status = sdk2.parameters.String("Result")
                result_pr = sdk2.parameters.Integer("Arcadia PR")

    @staticmethod
    def _get_pr_num(out):
        if ARCANUM_REVIEW_BASE not in out:
            raise ValueError("invalid output from pr create: {}".format(out))
        pr_num = out.split(ARCANUM_REVIEW_BASE)[1].strip()
        return int(pr_num)

    def on_execute(self):
        arc = self._arc()
        ticket = str(self.Parameters.ticket)
        config_dir = path.join("market/sre/conf/deplate/values", str(self.Parameters.root_abc).lower(),
                               str(self.Parameters.environment).lower())
        config_path = path.join(config_dir, "{}.yaml".format(self.Parameters.service_name))

        with arc.mount_path("", "trunk", fetch_all=False) as mp:
            arc.checkout(mp, branch=ticket, create_branch=True)

            try:
                makedirs(path.join(mp, config_dir))
            except OSError as err:
                if not "File exists" in err:
                    raise

            file_path = path.join(mp, config_path)
            if path.exists(file_path):
                self.Parameters.result_pr = 0
                self.Parameters.result_status = Status.NOOP
                self.set_info("Config {} already exists.".format(config_path))
                return

            cmd = [
                "values",
                "generate",
                "--name", str(self.Parameters.app_name),
                "--abc", str(self.Parameters.abc),
                "--environment", str(self.Parameters.environment),
                "--cpu", str(self.Parameters.cpu),
                "--ram", str(self.Parameters.ram),
                "--{}".format(self.Parameters.disk_type), str(self.Parameters.disk_size),
                "--sas-pods", str(self.Parameters.sas_pods),
                "--vla-pods", str(self.Parameters.vla_pods),
                "--man-pods", str(self.Parameters.man_pods),
                "--network-macro", str(self.Parameters.network_macro),
                "--exp3-consumer", str(self.Parameters.exp3_consumer),
                "--exp3-secret-id", str(self.Parameters.exp3_secret_id),
                "--exp3-secret-version", str(self.Parameters.exp3_secret_version),
                "--exp3-secret-alias", str(self.Parameters.exp3_secret_alias)
            ]
            if self.Parameters.app_command:
                cmd.extend(["--command", str(self.Parameters.app_command)])

            if str(self.Parameters.environment).lower() != 'unstable':
                cmd.append("--enable-infra-push-client")

            cmd.extend(["--template-root", str(self.Parameters.template_root)])
            for modifier in self.Parameters.template_modifiers:
                cmd.extend(["--template-modifiers", modifier])
            for tool in self.Parameters.template_tools:
                cmd.extend(["--template-tools", tool])

            cmd.append(file_path)
            self._run_command(cmd, mp)

            arc.add(mp, file_path)
            commit_message = "{}: Add new service: {}".format(ticket, self.Parameters.service_name)
            arc.commit(mp, commit_message)

            arc.push(mp, upstream="users/{}/{}-deplate-{}".format(self.Parameters.arc_user, ticket, self.Parameters.service_name), force=True)

            pr_num = self._get_pr_num(
                arc.pr_create(mp, publish=False, auto=True, message=commit_message, no_commits=True))
            self.Parameters.result_pr = pr_num
            self.Parameters.result_status = Status.COMMITTED
            self.set_info(
                'Arcanum <a href="{}{}">PR:{}</a> for new Deplate config was created.'.format(ARCANUM_REVIEW_BASE,
                                                                                              pr_num, pr_num),
                do_escape=False)
