# coding=utf-8
from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.projects.abc.client import AbcClient
from sandbox.common.errors import TaskFailure
from datetime import datetime, timedelta
import logging

class MarketDispenserReqTicketTagger(sdk2.Task):
    """
       Add 'market' tag to the 'our' DISPENSERREQ tickets
       where any of issue.followers in (abc.meta_market.members)
       and issue.queue = 'DISPENSERREQ'
    """

    class Parameters(sdk2.Task.Parameters):
        STARTREK_TOKEN_KEY = "STARTREK_TOKEN"
        ABC_TOKEN_KEY = "ABC_TOKEN"

        _ = sdk2.parameters.Info("Документация: https://st.yandex-team.ru/CSADMIN-34604")
        tasks_queue = sdk2.parameters.String("Очередь в Трекере", default="DISPENSERREQ", required=True)
        market_default_tag = sdk2.parameters.String("Назначаемый тег", default="market", required=True)
        abc_service_slug = sdk2.parameters.String("Имя сервиса в ABC", default="meta_market", required=True)
        abc_role_scope = sdk2.parameters.String("Роль в сервисе на ABC", default="quotas_management", required=True)
        yav_secret = sdk2.parameters.YavSecret("Секрет в YaV с токенами", required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4'])
        ]

    def on_execute(self):
        from startrek_client import Startrek

        st_oauth_token = self.Parameters.yav_secret.data()[self.Parameters.STARTREK_TOKEN_KEY]
        abc_oauth_token = self.Parameters.yav_secret.data()[self.Parameters.ABC_TOKEN_KEY]
        market_default_tag = self.Parameters.market_default_tag
        queue = self.Parameters.tasks_queue
        service_slug = self.Parameters.abc_service_slug
        role_scope = self.Parameters.abc_role_scope

        st = Startrek(useragent="sandbox-market-tracker-ticket-tagger", token=st_oauth_token)
        # Get quota_managers
        quota_managers = self._get_actual_members(abc_oauth_token, service_slug, role_scope, include_robots=False)

        # Create filter for issues
        start_date = self._get_start_date(datetime.now())
        st_filter = {
            "queue": queue,
            "created": {
                "from": start_date,
            },
            "followers": quota_managers,
        }

        # Get issues
        issues = st.issues.find(filter=st_filter)
        logging.info("Processing {} issues".format(len(issues)))
        # Process issues
        for issue in issues:
            logging.info("Processing issue {}".format(issue.key))
            users = map(lambda a: a.id, issue.followers)
            tags_added = False
            for user in users:
                if user in quota_managers and market_default_tag not in issue.tags:
                    issue.tags.append(market_default_tag)
                    tags_added = True
            if tags_added:
                issue.update(tags=issue.tags)
                logging.info("issue {} updated! issue tags {}".format(issue.key, issue.tags))

    @staticmethod
    def _get_actual_members(abc_token, service, role_scope, include_robots):
        abc = AbcClient(abc_token)
        service_info = abc.get_service_info_by_slug(service)

        # is_robot: True - only robots, False - exclude robots, None = both
        is_robot_map = {True: None, False: False}

        members = abc.get_people_from_service(
            service_id=service_info["id"],
            role_scope=role_scope,
            is_robot=is_robot_map[include_robots]
        )

        if not members:
            raise TaskFailure(
                "Can't get members of an ABC service: {}/{}".format(
                    service,
                    role_scope,
                )
            )
        return sorted(members)

    @staticmethod
    def _get_start_date(now):
        # returns a string formatted as YYYY-MM-DD, as required in ST filter
        DEFAULT_DAYS_OLD = 3
        return (now - timedelta(days=DEFAULT_DAYS_OLD)).strftime('%Y-%m-%d')

