# coding=utf-8

import logging
import os
import os.path
import tarfile
import tempfile
import shutil

import sandbox.common.types.resource as ctr

from sandbox import sdk2
from sandbox.projects.market.resources.sre import MARKET_DOWNTIMER
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.common.types import client as ctc


class MarketDowntimerDumpVLA(sdk2.Resource):
    """ VLA """


class MarketDowntimerDumpIVA(sdk2.Resource):
    """ IVA """


class MarketDowntimerDumpMAN(sdk2.Resource):
    """ MAN """


class MarketDowntimerDumpSAS(sdk2.Resource):
    """ SAS """


class MarketDowntimerExec(sdk2.Task):
    """Задача управления процссами дауйнтама сервисов во время учений. Создание dump, установка даунтайма и его удаление"""

    class TinyRequirements(sdk2.Requirements):
        disk_space = 1024
        cores = 1
        ram = 2048
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC)

        # disable all caches by default (make it multislot-aware)
        class Caches(sdk2.Requirements.Caches):
            pass

    DOWNTIMER_BIN = 'downtimer'

    class Parameters(sdk2.Task.Parameters):
        downtimer_dc = sdk2.parameters.String(
            'Имя ДЦ для даунтайма',
            default='vla'
        )
        downtimer_time = sdk2.parameters.String(
            'Время дайнтайма',
            default='24h'
        )
        downtimer_action = sdk2.parameters.String(
            'Выполняемое действие для утилиты',
            default='dump'
        )
        arcadia_conf_path = sdk2.parameters.String(
            'Путь к конфигам относительно корня Аркадии',
            default='market/sre/tools/downtimer/conf'
        )
        downtimer_conf_name = sdk2.parameters.String(
            'Имя конфига',
            default='market.yaml'
        )
        market_downtimer_resource = sdk2.parameters.LastReleasedResource(
            'Executable market downtimer',
            resource_type=MARKET_DOWNTIMER,
            state=ctr.State.READY,
            attrs={'released': 'stable'},
            required=True
        )
        tokens = sdk2.parameters.YavSecret(
            "YAV secret identifier (with optional version)",
            default="sec-01fxstyf20xpccbbe8f9zm73pc"
        )

    def get_resource(self, dc):
        dc_mapping = {
            "vla": MarketDowntimerDumpVLA,
            "sas": MarketDowntimerDumpSAS,
            "man": MarketDowntimerDumpMAN,
            "iva": MarketDowntimerDumpIVA
        }
        try:
            resource = dc_mapping[dc]
        except KeyError:
            raise RuntimeError("Unknown system: %s, supported: %s" %
                               (dc, list(dc_mapping.keys())))
        return resource

    def _get_config(self, tempdir):
        arcadia_conf_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, self.Parameters.arcadia_conf_path)
        return Arcadia.export(url=arcadia_conf_url, path=os.path.join(tempdir, 'conf/'))

    def get_action(self, action):
        action_mapping = {
            "dump": self._dump_action,
            "set": self._set_action,
            "remove": self._remove_action
        }
        try:
            func = action_mapping[action]
        except KeyError:
            raise RuntimeError("Unknown action: %s, supported: %s" %
                               (action, list(action_mapping.keys())))
        return func

    def _run_command(self, tempdir):
        tokens = self.Parameters.tokens.data()
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger(self.DOWNTIMER_BIN)) as pl:
            command_env = os.environ.copy()
            command_env['YDOWNTIMER_OAUTH'] = tokens["YDOWNTIMER_OAUTH"]
            login = tokens["YDOWNTIMER_LOGIN"]
            command = [os.path.join(tempdir, self.DOWNTIMER_BIN),
                       os.path.join(self._get_config(tempdir), self.Parameters.downtimer_conf_name),
                       self.Parameters.downtimer_action,
                       self.Parameters.downtimer_dc,
                       self.Parameters.downtimer_time,
                       login]
            process = sp.Popen(command, stdout=pl.stdout, stderr=sp.STDOUT, env=command_env)
            process.wait()
            pl.raise_for_status(process)

    def _dump_action(self, tempdir):
        resource = self.get_resource(self.Parameters.downtimer_dc)(self, "market downtimer dump", "dump.yaml")
        data = sdk2.ResourceData(resource)
        self._run_command(tempdir)
        data.ready()

    def _set_action(self, tempdir):
        market_downtimer_dump_resource = self.get_resource(self.Parameters.downtimer_dc).find(
            state=ctr.State.READY).first()
        data = sdk2.ResourceData(market_downtimer_dump_resource)
        shutil.copy(str(data.path), str(os.getcwd()))
        self._run_command(tempdir)

    def _remove_action(self, tempdir):
        self._run_command(tempdir)

    def on_execute(self):
        tempdir = tempfile.gettempdir()

        # Extract MARKET_DOWNTIMER
        resource_data = sdk2.ResourceData(self.Parameters.market_downtimer_resource)
        with tarfile.open(str(resource_data.path), 'r:gz') as tar:
            tar.extractall(path=tempdir)

        self.get_action(self.Parameters.downtimer_action)(tempdir)
