# -*- coding: utf-8 -*-

import logging
import os
import re
from contextlib import contextmanager

from sandbox import sdk2
from sandbox.common.errors import TaskFailure
from sandbox.sdk2.vcs.svn import Arcadia
from sandbox.sdk2.helpers import subprocess as sp


@contextmanager
def _chdir(path):
    curdir = os.path.abspath(os.curdir)
    os.chdir(path)
    yield
    os.chdir(curdir)


class MarketGencfgGroupChangeHosts(sdk2.Task):
    """Task to change hosts in gencfg group"""

    class Parameters(sdk2.Task.Parameters):
        arcadia_user = "zomb-sandbox-rw"
        group_name = sdk2.parameters.String('GenCfg group name to change hosts', required=True)
        hosts_count = sdk2.parameters.Integer('Target number of hosts in GenCfg group', required=True)
        startrek_ticket = sdk2.parameters.String("Startrek ticket", required=False)
        with sdk2.parameters.Output:
            commit_url = sdk2.parameters.String("Commit URL", default="", required=True)

    def on_execute(self):
        self._prepare_env()
        current_hosts_count = self._get_current_hosts_count()
        if current_hosts_count == self.Parameters.hosts_count:
            logging.info('Group %s already has %d instances', self.Parameters.group_name, self.Parameters.hosts_count)
            self.Parameters.commit_url = "Not changed"
        else:
            self._add_hosts(self.Parameters.hosts_count - current_hosts_count)
            self._run_checks()
            self._commit_changes()
        logging.info('Changed: %s', self.Parameters.commit_url)

    def _prepare_env(self):
        logging.info('Preparing an environment for GenCfg')

        gencfg_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, 'gencfg')
        self.gencfg_path = Arcadia.checkout(gencfg_url, str(self.path('gencfg')))
        logging.info('Arcadia is checkouted to %s', self.gencfg_path)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('install')) as pl:
            process = sp.Popen(
                ['bash', './install.sh'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('update')) as pl:
            process = sp.Popen(
                ['bash', './update.sh'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

    def _get_current_hosts_count(self):
        logging.info('Get current number of hosts')

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('get_current_hosts_count')) as pl:
            process = sp.Popen(
                ['/skynet/python/bin/python', './utils/common/show_group_hosts.py',
                    '-g', self.Parameters.group_name],
                cwd=self.gencfg_path,
                stdout=sp.PIPE,
                stderr=pl.stdout
            )
            process.wait()

        if process.returncode:
            raise TaskFailure('Failed to get hosts for group')

        out = process.communicate()

        hosts = out[0].split('\n')[0].split(',')
        if hosts == ['']:
            return 0
        for host in hosts:
            if not host.endswith('.search.yandex.net'):
                raise TaskFailure('Bad answer from show_group_hosts.py: "{}"'.format(out[0]))
        return len(hosts)

    def _add_hosts(self, instances):
        logging.info('Adding hosts to group')

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('add_hosts')) as pl:
            process = sp.Popen(
                ['/skynet/python/bin/python', './optimizers/dynamic/recluster.py',
                    '-g', self.Parameters.group_name,
                    '--instances', str(instances),
                    '-y'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()

        if process.returncode:
            raise TaskFailure('Failed to add hosts to group')

    def _run_checks(self):
        logging.info('Running checks')
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger('run_checks')) as pl:
            process = sp.Popen(
                ['bash', './gen.sh', 'run_checks'],
                cwd=self.gencfg_path,
                stdout=pl.stdout,
                stderr=sp.STDOUT
            )
            process.wait()
            pl.raise_for_status(process)

    def _commit_changes(self):
        logging.info('Commiting changes')

        groups_path = 'db'

        if self.Parameters.startrek_ticket:
            ticket = self.Parameters.startrek_ticket + " "
        else:
            ticket = ""

        with _chdir(os.path.join(self.gencfg_path, groups_path)):
            logging.info('Svn status: %s', Arcadia.status(path='.'))
            logging.info('Svn diff: \n%s', Arcadia.diff('.'))

            output = Arcadia.commit(path='.',
                                    message='{}Add hosts to group {}'.format(ticket, self.Parameters.group_name),
                                    user=self.Parameters.arcadia_user)
            if not output:
                raise TaskFailure('Nothing has been commited')
            revision = re.findall(r'Committed revision (\d+)\.', output)
            if revision:
                self.Parameters.commit_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])

        logging.info('Changes have been commited')
