# coding=utf-8
import datetime
import json
import logging
import requests

from sandbox import sdk2
from sandbox.sandboxsdk.environments import PipEnvironment


class MarketHwWatcherTicketsHandler(sdk2.Task):
    """Обработка тикетов от HW Watcher."""

    class Parameters(sdk2.Task.Parameters):
        _ = sdk2.parameters.Info('Документация: https://nda.ya.ru/t/5x7RxE9E3W56oc')
        tasks_queue = sdk2.parameters.String('Очередь в Трекере', default='CSADMIN')
        tasks_author = sdk2.parameters.List('Автор тикета (имя пользователя)', default=['robot-automon'])
        stale_ticket_days = sdk2.parameters.Integer(
            'Количество дней, после которых тикет будет считаться без активности', default=2)
        secret = sdk2.parameters.YavSecret('YAV секрет с OAuth токенами', required=True)

    class Requirements(sdk2.Task.Requirements):
        environments = [
            PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
        ]

    def get_st_api(self):
        """Получение объекта Startrek API."""
        from startrek_client import Startrek

        st_token = self._secret.data()['startrek_token']
        return Startrek(
            useragent='sandbox-market-hwwatcher-tickets-handler',
            token=st_token)

    def get_issues_for_status(self, st, status):
        """Получение списка тикетов для указанного статуса."""
        # Component CS.HW_watcher
        component_hwwatcher_id = 33505
        st_filter = {
            'queue': self.Parameters.tasks_queue,
            'author': self.Parameters.tasks_author,
            'components': [component_hwwatcher_id],
            'status': status
        }
        return st.issues.find(filter=st_filter)

    def get_bot_url(self, issue):
        """Получение URL из тикета."""
        description = []
        if issue.description:
            description.extend(issue.description.split('\n'))
        for line in description:
            if line.startswith('https://bot.yandex-team.ru/api/v3/'):
                bot_url = line.replace('&amp;', '&')
                logging.info('Find URL %s', bot_url)
                return bot_url
        return None

    def create_bot_issue(self, url):
        """Выполнение запроса на URL и получение ключа созданного тикета."""
        bot_token = self._secret.data()['bot_token']
        headers = {
            'Authorization': 'OAuth {}'.format(bot_token),
            'Content-Type': 'application/json'
        }
        r = requests.get(url, headers=headers)
        r.raise_for_status()
        resp = json.loads(r.text)
        bot_issue = resp['result']['StNum']
        logging.info('Create issue %s', bot_issue)
        return bot_issue

    def get_potential_assignee(self, issue):
        """Определение исполнителя."""
        assignee = None
        # Кто последний менял статус, тот и будем исполнителем
        for item in issue.changelog:
            if item.type == 'IssueWorkflow':
                for field in item.fields:
                    if field['field'].id == 'status' and field['to'].key == 'selectedForDev':
                        assignee = item.updatedBy
        logging.info('Assignee is %s', assignee.login)
        return assignee

    def process_new_issue(self, issue):
        """Обработка новых тикетов."""
        # Берем тикет в работу для того, чтобы были доступны большинство переходов
        issue.transitions['start_progress'].execute()
        bot_url = self.get_bot_url(issue)
        if bot_url:
            try:
                bot_issue = self.create_bot_issue(bot_url)
                issue.links.create(issue=bot_issue, relationship='depends on')
                assignee = self.get_potential_assignee(issue)
                transition = issue.transitions['need_info']
                transition.execute(comment='Связанный тикет {}.'.format(bot_issue), assignee=assignee)
                return
            except Exception as e:
                comment = 'Не удалось создать связанный тикет, требуется обработать вручную.'
                logging.error(e)
        else:
            comment = 'Не удалось найти ссылку в тикете, требуется обработать вручную.'
        logging.info('Return issue in state Open')
        transition = issue.transitions['stop_progress']
        transition.execute(comment=comment)

    def check_is_issue_stale(self, issue):
        """Проверка, что в тикете есть активность."""
        last_action_time_str = datetime.datetime.utcfromtimestamp(0)
        # Получаем время последнего изменения тикета
        for item in issue.changelog:
            last_action_time_str = item.updatedAt
        last_action_time = datetime.datetime.strptime(last_action_time_str, '%Y-%m-%dT%H:%M:%S.%f+0000')
        no_action_time = datetime.datetime.utcnow() - last_action_time
        stale_ticket_days = datetime.timedelta(days=self.Parameters.stale_ticket_days)
        if no_action_time > stale_ticket_days:
            issue.comments.create(text='Тикет подвис, вы обо мне не забыли?', summonees=[issue.assignee])

    def check_is_server_up(self, issue):
        fqdn = issue.summary.split(' ')[-1]
        r = requests.get(
            'https://api.wall-e.yandex-team.ru/v1/hosts/{}'.format(fqdn),
            params={'fields': 'health.check_statuses'})
        r.raise_for_status()
        resp = json.loads(r.text)
        unreachable_status = resp['health']['check_statuses']['unreachable']
        ssh_status = resp['health']['check_statuses']['ssh']
        logging.info('unreachable: {}, ssh: {}'.format(unreachable_status, ssh_status))
        if unreachable_status == 'passed' and ssh_status == 'passed':
            return True
        return False

    def process_issue_in_work(self, issue):
        """Обработка тикетов в работе."""
        links = False
        # Если хоть один связанный тикет не закрыт - выходим
        for link in issue.links:
            links = True
            if link.object.status.key != 'closed':
                logging.info('Find not closed linked issue %s', link.object.key)
                return
        if links:
            if self.check_is_server_up(issue):
                logging.info('Close ticket %s', issue.key)
                transition = issue.transitions['close']
                transition.execute(comment='Связанный тикет закрыт', resolution='fixed')
            else:
                logging.info('Linked issues is closed, but server is down.')
                issue.transitions['inProgress'].execute()
                issue.comments.create(text='Связанные тикеты закрыты, на сервер недоступен. Нужно разобраться.', summonees=[issue.assignee])
        self.check_is_issue_stale(issue)

    def on_execute(self):
        self._secret = self.Parameters.secret

        st = self.get_st_api()

        # Обрабатываем тикеты в работе. Если все связанные закрыты - закрываем и этот.
        logging.info('Start processing tickets in work')
        issues_in_work = self.get_issues_for_status(st, 'needInfo')
        for issue in issues_in_work:
            logging.info('Process ticket: %s', issue.key)
            self.process_issue_in_work(issue)

        # Обрабатываем новые тикеты. Создаем связанный тикет
        logging.info('Start processing new tickets')
        new_issues = self.get_issues_for_status(st, 'selectedForDev')
        for issue in new_issues:
            logging.info('Process ticket: %s', issue.key)
            self.process_new_issue(issue)
