# -*- coding: utf-8 -*-

import logging
from os import path

from sandbox import sdk2
from sandbox.common.errors import TaskError
from sandbox.common.types import client as ctc
from sandbox.projects.common.task_env import TinyRequirements
from sandbox.projects.common.vcs.arc import Arc
from sandbox.projects.market.resources.sre import MarketNewNannyServiceBinary
from sandbox.sdk2.helpers import subprocess as sp


class MarketNewNannyServiceRun(sdk2.Task):
    """
        Run new-nanny-service
    """

    class Requirements(TinyRequirements):
        disk_space = 1 * 1024
        client_tags = (ctc.Tag.MULTISLOT | ctc.Tag.GENERIC) & ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group("New Nanny Service", collapse=False) as deplate_block:
            new_nanny_service_binary = sdk2.parameters.Resource(
                'new-nanny-service binary',
                resource_type=MarketNewNannyServiceBinary,
                required=True,
            )
            program_args = sdk2.parameters.String('new-nanny-service args')
            new_nanny_service_token = sdk2.parameters.YavSecret(
                "new-nanny-service token",
                required=True
            )

    def on_create(self):
        resource = MarketNewNannyServiceBinary.find(attrs={"released": "stable"}).first()
        if resource:
            self.Parameters.new_nanny_service_binary = resource

    def _prepare_task(self):
        if not self.Context.new_nanny_service_path or not self.Context.new_nanny_service_token:
            data = sdk2.ResourceData(self.Parameters.new_nanny_service_binary)
            self.Context.new_nanny_service_path = path.join(str(data.path), "new-nanny-service")

            key = 'token'
            if self.Parameters.new_nanny_service_token.default_key:
                key = self.Parameters.new_nanny_service_token.default_key

            self.Context.new_nanny_service_token = self.Parameters.new_nanny_service_token.data()[key]

    def _run_command(self, args_string):
        self._prepare_task()

        env = {
            "NANNY_SANDBOX_TOKEN": self.Context.new_nanny_service_token,
        }

        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("new-nanny-service")) as pl:
            sp.check_call("{} {}".format(self.Context.new_nanny_service_path, args_string), shell=True, stdout=pl.stdout,
                          stderr=pl.stderr, env=env)

    def on_execute(self):
        self._run_command(self.Parameters.program_args)
