# -*- coding: utf-8 -*-
"""This task removes gencfg groups via utils/market/update_igroups.py"""

import os
import logging

from sandbox import common
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk.process import run_process
from sandbox.sdk2.vcs.svn import Svn
from sandbox.projects.common import utils
from sandbox.projects.common.gencfg.task import IGencfgBuildTask
from contextlib import contextmanager


@contextmanager
def _chdir(path):
    curdir = os.path.abspath(os.curdir)
    os.chdir(path)
    yield
    os.chdir(curdir)


class Groups(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'groups'
    description = 'Gencfg groups to remove'
    default_value = []


class CommitMessage(parameters.SandboxStringParameter):
    name = 'commit_message'
    description = 'Commit message'
    required = True


class RemoveGencfgGroups(IGencfgBuildTask):
    type = 'MARKET_REMOVE_GENCFG_GROUPS'

    input_parameters = [Groups, CommitMessage]

    def on_execute(self):
        self._prepare_env()
        present_groups = set()
        for group in utils.get_or_default(self.ctx, Groups):
            if group in present_groups:
                continue
            if not self._check_exists(group):
                logging.info("Group %s doesn't exist", group)
                continue
            present_groups.add(group)
        if not present_groups:
            logging.info("Nothing to remove.")
            return
        self._remove_groups(present_groups)
        self._run_checks()
        self._commit_changes()

    def _prepare_env(self):
        logging.info('Preparing an environment for GenCfg')
        self.clone_and_install_generator(precalc_caches=True, load_cache_resource=True)

    def _remove_groups(self, present_groups):
        logging.info('Removing groups')

        failed = False

        for group in present_groups:
            parts = group.split("_")
            if parts[1] != "MARKET":
                logging.error("Group %s doesn't belong to MARKET. Unsupported")
                failed = True

            process = run_process(
                ' '.join(['/skynet/python/bin/python', './utils/common/update_igroups.py', '-a', 'remove', '-g', group]),
                work_dir=self.get_gencfg_path(),
                log_prefix='remove_group_{}'.format(group),
                check=False,
            )

            if process.returncode:
                failed = True
                self.set_info(process.saved_cmd)
                with open(process.stdout_path) as fp:
                    self.set_info(fp.read(), do_escape=False)

        if failed:
            raise common.errors.TaskFailure('Failed to remove groups')

    def _run_checks(self):
        logging.info('Running checks')
        self.build_generator(create_resources=False, run_only_checks=True)

    def _check_exists(self, group):
        process = run_process(
            ' '.join(['/skynet/python/bin/python', './utils/common/has_group.py', group]),
            work_dir=self.get_gencfg_path(),
            outs_to_pipe=True,
            log_prefix='check_group_exists',
            check=False,
        )

        if process.returncode:
            self.set_info(process.saved_cmd)
            raise common.errors.TaskFailure('Failed to check if group exists')

        result_str = " ".join(process.stdout.readlines()).strip()
        return result_str == 'True'

    def _commit_changes(self):
        logging.info('Commiting changes')

        groups_path = 'db'

        with _chdir(os.path.join(self.get_gencfg_path(), groups_path)):
            logging.info('Svn status: %s', Svn.status(path='.'))
            output = Svn.commit(path='.', message=self.ctx.get(CommitMessage.name))
            if not output:
                raise common.errors.TaskFailure('Nothing has been commited')

        logging.info('Changes have been commited')


__Task__ = RemoveGencfgGroups
