# coding: utf-8

import os
import datetime
import logging

from tempfile import mkdtemp

from sandbox.sandboxsdk.task import SandboxTask
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.vcs.git import Git
from sandbox.sandboxsdk.ssh import Key


VAULT_SSH = "csadmin-ssh"
VAULT_OWNER = "MARKETSRE"
_REPORT_NAME = 'ExtData/MarketServerCount'
_REPORT_SCALE = 'daily'
_SERVER_STATBOX = 'stat.yandex-team.ru'
_REPO_URL = 'git@github.yandex-team.ru:asimakov/inventory-data.git'
_FILE_FOR_COUNT = 'out_MARKET.all.sorted.txt'


class MarketServerCountToStat(SandboxTask):
    """
        Считаем количество серверов в проекте Market
    """
    type = 'MARKET_SERVER_COUNT_TO_STAT'

    environment = (
        PipEnvironment('python-statface-client', '0.65.0', use_wheel=False),
    )

    def _git_clone(self, repo):
        logging.info("Cloning repo {}".format(repo))
        with Key(self, VAULT_OWNER, VAULT_SSH):
            dirname = mkdtemp()
            git = Git(repo)
            git.clone(dirname, "master")
            return dirname

    def on_execute(self):
        logging.debug("cloning repo %s" % (_REPO_URL))
        inventory_data_dir = self._git_clone(_REPO_URL)
        # -1 because comments in first line
        num_lines = sum(1 for line in open(os.path.join(inventory_data_dir, _FILE_FOR_COUNT)))-1
        logging.debug("num_lines = %d" % (num_lines))

        self.statfaceUpload(num_lines)

    def statfaceUpload(self, count):
        from statface_client import StatfaceClient
        stat_user = "csadmin"
        stat_token = self.get_vault_data('csadmin-statbox')

        stat = StatfaceClient(stat_user, '', _SERVER_STATBOX, '', stat_token)
        report = stat.get_report(_REPORT_NAME)
        data = dict()
        data['fielddate'] = self._get_date(_REPORT_SCALE)
        data['count'] = count

        logging.debug('upload data: {}'.format(data))
        report.upload_data(_REPORT_SCALE, [data])

    def _get_date(self, scale):
        fmt = {
            'daily': '%Y-%m-%d',
            'hourly': '%Y-%m-%d %H:00:00',
            'minutely': '%Y-%m-%d %H:%M:00',
        }.get(scale, None)
        if fmt is None:
            logging.fatal('Invalid scale value')
        return datetime.datetime.now().strftime(fmt)


__Task__ = MarketServerCountToStat
