# coding=utf-8

import logging
import os
import re

from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.sdk2.vcs.svn import Arcadia, SvnError


class MarketWmsBalancerConfigCommit(sdk2.Task):
    """Commit new Balancer Config to Arcadia. For Market balancers only.
    """

    class Requirements(task_env.BuildLinuxRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        ARCADIA_USER = "zomb-sandbox-rw"
        SRE_CONF_PATH = "market/sre/conf"

        startrek_ticket = sdk2.parameters.String("Startrek ticket", required=True)
        pre_configs_dir = sdk2.parameters.String(
            "Directory with precommitted configs in market/sre/conf", default="wms-balancers-pipeline", required=True)
        with sdk2.parameters.Output:
            arcadia_url = sdk2.parameters.String("Commit URL", default="")

    def on_execute(self):
        sre_conf_path = self._checkout_arcadia_path(self.Parameters.SRE_CONF_PATH)
        pre_configs_dir = os.path.join(
            sre_conf_path,
            str(self.Parameters.pre_configs_dir),
            str(self.Parameters.startrek_ticket)
        )
        if not os.path.isdir(pre_configs_dir):
            raise Exception("Directory {} not exists".format(pre_configs_dir))

        for balancer_type in ("haproxy", "nginx"):
            for env in ("test", "prod"):
                src_path = os.path.join(pre_configs_dir, balancer_type, env)
                dst_path = os.path.join(
                    sre_conf_path,
                    "slb-{0}/{1}/etc/{0}/values-available".format(balancer_type, env)
                )
                self._move_configs(src_path, dst_path)
        Arcadia.delete(pre_configs_dir)
        self._commit_changes(sre_conf_path)

    def _checkout_arcadia_path(self, path):
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, path)
        arcadia_path = Arcadia.checkout(arcadia_url, str(self.path("arcadia")))
        logging.info("Arcadia is checkouted to {}".format(arcadia_path))
        return arcadia_path

    def _move_configs(self, src_dir, dst_dir):
        if not os.path.isdir(src_dir):
            logging.info("Path %s not found. Skip it", src_dir)
            return
        for filename in os.listdir(src_dir):
            file = os.path.join(src_dir, filename)
            if os.path.isfile(file):
                logging.info("Copy file %s to %s/", file, dst_dir)
                os.rename(file, os.path.join(dst_dir, filename))

    def _commit_changes(self, arcadia_path):
        Arcadia.add(arcadia_path, force=True)
        logging.info("Commiting changes to Arcadia")
        logging.info('Svn status: \n%s', Arcadia.status(path=arcadia_path))
        logging.info('Svn diff: \n%s', Arcadia.diff(arcadia_path))

        try:
            result = Arcadia.commit(path=arcadia_path,
                                    message="{} Commited new balancer configs".format(self.Parameters.startrek_ticket),
                                    user=self.Parameters.ARCADIA_USER)
            logging.info("Commit result: %s", result)
            revision = re.findall(r'Committed revision (\d+)\.', result)
            if revision:
                self.Parameters.arcadia_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            logging.error(e.message)
            # For Arcadia reviews
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            if review:
                self.Parameters.arcadia_url = "https://a.yandex-team.ru/review/{}".format(review[0])
