# coding=utf-8

import logging
import os
import re

from sandbox import sdk2
from sandbox.projects.common import task_env
from sandbox.sdk2.vcs.svn import Arcadia, SvnError


class MarketWmsBalancerConfigPreCommit(sdk2.Task):
    """Pre commit new Balancer Config to Arcadia. For Market balancers only.
    """

    class Requirements(task_env.BuildLinuxRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        ARCADIA_USER = "zomb-sandbox-rw"
        SRE_CONF_PATH = "market/sre/conf"

        startrek_ticket = sdk2.parameters.String("Startrek ticket", required=True)
        haproxy_configs = sdk2.parameters.JSON("Haproxy configs. filename: content")
        nginx_configs = sdk2.parameters.JSON("Nginx configs. filename: content")
        pre_configs_dir = sdk2.parameters.String(
            "Directory with precommitted configs in market/sre/conf", default="wms-balancers-pipeline", required=True)
        with sdk2.parameters.Output:
            arcadia_url = sdk2.parameters.String("Commit URL", default="")

    def on_execute(self):
        logging.info(self.Parameters.haproxy_configs)
        logging.info(self.Parameters.nginx_configs)

        arcadia_path = self._checkout_arcadia_path(os.path.join(
            str(self.Parameters.SRE_CONF_PATH),
            str(self.Parameters.pre_configs_dir)
        ))
        pipeline_ticket_dir = os.path.join(arcadia_path, str(self.Parameters.startrek_ticket))
        if os.path.isdir(pipeline_ticket_dir):
            raise Exception("Directory {} already exists".format(pipeline_ticket_dir))

        pipeline_haproxy_conf_dir = os.path.join(pipeline_ticket_dir, "haproxy")
        self._save_configs(pipeline_haproxy_conf_dir, self.Parameters.haproxy_configs)
        pipeline_nginx_conf_dir = os.path.join(pipeline_ticket_dir, "nginx")
        self._save_configs(pipeline_nginx_conf_dir, self.Parameters.nginx_configs)
        self._commit_changes(pipeline_ticket_dir)

    def _checkout_arcadia_path(self, path):
        arcadia_url = os.path.join(Arcadia.ARCADIA_TRUNK_URL, path)
        path = Arcadia.checkout(arcadia_url, str(self.path("arcadia")))
        logging.info("Arcadia is checkouted to {}".format(path))
        return path

    def _save_configs(self, path, configs):
        for filename, content in configs.items():
            filepath = os.path.join(path, filename)
            if not os.path.exists(os.path.dirname(filepath)):
                os.makedirs(os.path.dirname(filepath))
            if content[-1] != "\n":
                content += "\n"
            with open(filepath, "w") as fd:
                fd.write(content)

    def _commit_changes(self, arcadia_path):
        Arcadia.add(arcadia_path)
        logging.info("Commiting changes to Arcadia")
        logging.info('Svn status: \n%s', Arcadia.status(path=arcadia_path))
        logging.info('Svn diff: \n%s', Arcadia.diff(arcadia_path))

        try:
            result = Arcadia.commit(path=arcadia_path,
                                    message="{} Commited new balancer configs".format(self.Parameters.startrek_ticket),
                                    user=self.Parameters.ARCADIA_USER)
            logging.info("Commit result: %s", result)
            revision = re.findall(r'Committed revision (\d+)\.', result)
            if revision:
                self.Parameters.arcadia_url = "https://a.yandex-team.ru/arc/commit/{}".format(revision[0])
        except SvnError as e:
            logging.error(e.message)
            # For Arcadia reviews
            review = re.findall(r'https://a\.yandex-team\.ru/review/(\d+)', e.message)
            if review:
                self.Parameters.arcadia_url = "https://a.yandex-team.ru/review/{}".format(review[0])
