# -*- coding: utf-8 -*-

import logging

from sandbox.projects.market.sre.MarketYaPackageCommon import tsum

from sandbox import sdk2
from sandbox.common.types import resource as ctr
from sandbox.projects import resource_types
from sandbox.projects.common import binary_task
from sandbox.projects.common import task_env
from sandbox.projects.common.build.YaPackage2 import YaPackage2
from sandbox.projects.common.build.ya_package_config import consts as ya_package_consts
from sandbox.projects.common.nanny import nanny
from sandbox.projects.common.ya_deploy import release_integration
import sandbox.projects.common.build.parameters as build_parameters
import sandbox.projects.common.constants as consts


class ResourceTypeParameter(sdk2.parameters.String):
    name = "resource_type"
    description = "Resource type"
    required = True

    @classmethod
    def get_custom_parameters(cls):
        return dict(
            values=sorted(((rt.name, rt.name) for rt in resource_types.AbstractResource if "MARKET" in rt.name)))


class TsumSendReportParameter(sdk2.parameters.Bool):
    name = "tsum_send_report"
    description = "Send report to TSUM"
    required = False
    default_value = False


class MarketYaPackage2(YaPackage2):
    """
        YaPackage2 Маркета c переопределенными значениями параметров.
    """

    class Requirements(task_env.BuildLinuxRequirements):
        pass

    class Parameters(YaPackage2.Parameters):
        use_aapi_fuse = build_parameters.UseArcadiaApiFuse(default_value=True)
        use_arc_instead_of_aapi = build_parameters.UseArcInsteadOfArcadiaApi(default_value=False)
        aapi_fallback = build_parameters.AllowArcadiaApiFallback(default_value=True)

        packages = sdk2.parameters.String("Package path", required=True)
        package_type = build_parameters.PackageType(default_value=ya_package_consts.PackageType.TARBALL.value)
        compress_package_archive = build_parameters.CompressPackageArchive(default_value=True)

        resource_type = ResourceTypeParameter()
        release_to_nanny = build_parameters.ReleaseToNanny(default_value=True)
        release_to_ya_deploy = release_integration.ReleaseToYaDeployParameter2(default_value=False)

        checkout = build_parameters.CheckoutParameter(default_value=True)
        build_system = build_parameters.BuildSystem(default_value=consts.SEMI_DISTBUILD_BUILD_SYSTEM)

        other_block = sdk2.parameters.Info("Market params")
        startrek_ticket_ids = nanny.StartrekTicketIdsParameter2("Startrek ticket ids")
        tsum_send_report = TsumSendReportParameter()
        resource_ttl = sdk2.parameters.String("Resource ttl", required=False, default="inf")

        ext_params = binary_task.binary_release_parameters(stable=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": self.type.name, "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_success(self, prev_status):
        if self.Parameters.tsum_send_report:
            for package_resource in self.Context.package_resources["resources"]:
                resource = sdk2.Resource[package_resource[0]]
                if resource.type.releasable:
                    logging.info("Calculate md5 checksums for resource %s", resource.path)
                    resource_abs_path = str(sdk2.ResourceData(resource).path)
                    tsum.send_report_to_tsum(resource_abs_path, str(resource.type), self.id, resource.id)

    def on_release(self, additional_parameters):
        YaPackage2.on_release(self, additional_parameters)

        ttl = self.Parameters.resource_ttl
        if not ttl:
            ttl = "inf"
        self.mark_released_resources(additional_parameters["release_status"], ttl)
