from io import BytesIO
import logging
import json
import os
import time
import hashlib
import tarfile

import requests

CHUNK_SIZE_BYTES = 1048576  # 1 MByte
RETRY_COUNT = 3
RETRY_PAUSE_SECONDS = 10
TSUM_API_HOSTS = [
    'https://tsum-agent-api-testing.market.yandex.net:4209',
    'https://tsum-agent-api.market.yandex.net:4209',
]

logger = logging.getLogger()


def send_report_to_tsum(archive, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id=None):
    headers = {'Content-type': 'application/json'}
    report_json = json.dumps(build_md5_report(archive, sandbox_resource_type, sandbox_ticket_num, sandbox_resource_id))

    for tsum_api_host in TSUM_API_HOSTS:
        for i in xrange(RETRY_COUNT):
            try:
                logging.info("Send report to TSUM host %s", tsum_api_host)

                report_url = tsum_api_host + '/agent/addResourceInfo'
                response = requests.post(url=report_url, data=report_json, headers=headers)

                response.raise_for_status()
                logging.info("Successfully sent report checksums to sandbox API")
                break
            except StandardError as e:
                logging.info("Error %s. Report body %s", str(e.message), report_json)

                if i == RETRY_COUNT - 1:
                    logging.fatal("Can't send report to tsum")
                else:
                    time.sleep(i * RETRY_PAUSE_SECONDS)


def build_md5_report(archive_path, sandbox_resource_type, sandbox_task_id, sandbox_resource_id):
    def get_md5(check_file):
        hash_md5 = hashlib.md5()

        while True:
            file_bytes = check_file.read(CHUNK_SIZE_BYTES)
            if file_bytes:
                hash_md5.update(file_bytes)
            else:
                break

        return hash_md5.hexdigest()

    logging.info("Calculate md5 checksums for %s", archive_path)
    checksums = []
    if ".uc" in archive_path:
        logger.info("Assuming that package %s was compressed with uc", archive_path)
        # IO class that does not call self.close() after succsessful exit of context manager
        class ConsistentBytesIO(BytesIO):
            # We don't want to close our BytesIO after leaving context manager, we want to do it mannualy using .close() method
            def __exit__(self, exc_type, exc_val, exc_tb):
                # If exception occured we will exit normaly
                if exc_type:
                    super(BytesIO, self).__exit__(exc_type, exc_val, exc_tb)

        decompressed_tar = ConsistentBytesIO()
        def _fopen(file, *args, **kwargs):
            if isinstance(file, ConsistentBytesIO):
                return file
            else:
                return open(file, *args, **kwargs)

        logger.debug("Decompressing with uc; tar -> ConsistentBytesIO")
        from library.python.compress import decompress
        decompress(archive_path, decompressed_tar, None, _fopen, _safe_get_cpu_count())
        logger.debug("Finished decompressing")

        logger.debug("Creating tar object from decompressed object (ConsistentBytesIO)")
        decompressed_tar.seek(0)  # Like if we just opened a file for read
        tar = tarfile.open(fileobj=decompressed_tar, mode="r:")
    else:
        logger.debug("Creating tar object from file %s", archive_path)
        tar = tarfile.open(archive_path)

    def extract_tar_members(tar_file):
        for member in tar_file.getmembers():
            try:
                res = tar_file.extractfile(member)
                yield res, member
            except KeyError:
                # ignore non existing symlinks
                if hasattr(member, "islnk") and member.islnk():
                    continue
                elif hasattr(member, "issym") and member.issym():
                    continue
                raise

    for sub_file, sub_member in extract_tar_members(tar):
        if sub_file:
            logger.debug("Calculating md5sum for %s", sub_member.name)
            checksums.append({
                "path": sub_member.name,
                "md5": get_md5(sub_file)
            })
            sub_file.close()

    logger.debug("Md5sum calculated successfully, closing tar")
    # If we used ConsistentBytesIO to create tar object, it will be closed inside tar.close() method
    tar.close()

    with open(archive_path) as tar_archive:
        md5_report = {
            "resource": {
                "resourceType": sandbox_resource_type,
                "resourceId": sandbox_resource_id,
                "taskId": sandbox_task_id
            },
            "md5": get_md5(tar_archive),
            "checksums": checksums
        }

    logging.info("MD5 checksums sending to tsum-sox api: %s" % str(md5_report))
    return md5_report


# https://stackoverflow.com/a/55423170
def _safe_get_cpu_count():
    cpu_count = 4
    try:
        cpu_count = len(os.sched_getaffinity(0))
        logger.debug("len(os.sched_getaffinity(0)) returned %s", cpu_count)
    except AttributeError:
        # If module os does not have sched_getaffinity function
        logger.warn("Could not get access to os.sched_getaffinity to get usable cpu count. Fallback to value 4")
    return cpu_count
