# coding: utf-8

import glob
import os
import tarfile

import sandbox.common.types.client as ctc

from sandbox.projects.juggler import resource_types
from sandbox.projects.common import utils
from sandbox.sandboxsdk import errors
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import process
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.channel import channel
from sandbox.sandboxsdk.environments import PipEnvironment
from sandbox.sdk2.vcs.git import Git


class JugglerAnsibleResource(parameters.LastReleasedResource):
    name = 'juggler_ansible_resource'
    description = 'Juggler Ansible resource'
    resource_type = resource_types.JUGGLER_ANSIBLE


class PlaybookRepo(parameters.SandboxGitUrlParameter):
    name = 'playbook_repo'
    description = 'Git repository with playbooks'
    required = True
    default_value = 'https://github.yandex-team.ru/cs-admin/ansible-juggler-configs.git'


class PlaybookPath(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'playbook_files'
    description = 'Playbook files'
    required = True
    default_value = ['services/*.yml', 'services-for-devs/*.yml']


class CreateTicket(parameters.SandboxBoolParameter):
    name = 'create_ticket'
    description = 'Create Startrek ticket on failure'
    default_value = 'CSADMIN'
    sub_fields = {
        'true': ['startrek_queue', 'ticket_tags', 'startrek_secret_name'],
    }


class StartrekQueue(parameters.SandboxStringParameter):
    name = 'startrek_queue'
    description = 'Startrek queue'
    default_value = 'CSADMIN'


class TicketTag(parameters.ListRepeater, parameters.SandboxStringParameter):
    name = 'ticket_tags'
    description = 'Ticket tags'
    default_value = ['cs_duty']


class SecretOwner(parameters.SandboxStringParameter):
    name = 'secret_owner'
    description = 'Vault secret owner'
    default_value = 'MARKETSRE'


class StartrekSecretName(parameters.SandboxStringParameter):
    name = 'startrek_secret_name'
    description = 'Vault secret name with Startrek token (startrek_csadmin_token)'
    ENV_VAR = 'STARTREK_TOKEN'


class ConductorSecretName(parameters.SandboxStringParameter):
    name = 'conductor_secret_name'
    description = 'Vault secret name with Conductor token (conductor_csadmin_token)'
    ENV_VAR = 'CONDUCTOR_TOKEN'


class NannySecretName(parameters.SandboxStringParameter):
    name = 'nanny_secret_name'
    description = 'Vault secret name with Nanny token (nanny_csadmin_token)'
    ENV_VAR = 'NANNY_TOKEN'


class RunJugglerMarketPlaybook(task.SandboxTask):
    """
    Apply Market's juggler playbooks.
    The task checks the repository with ansible-juggler playbooks.
    Finds the files and executes ansible-juggler against every of them.
    In case there are failed playbooks it creates Startrek ticket.
    """

    type = 'RUN_JUGGLER_MARKET_PLAYBOOK'

    execution_space = 100
    client_tags = ctc.Tag.LINUX_PRECISE

    tokens = [
        StartrekSecretName,
        ConductorSecretName,
        NannySecretName,
    ]

    input_parameters = [
        JugglerAnsibleResource,
        PlaybookRepo,
        PlaybookPath,
        CreateTicket,
        StartrekQueue,
        TicketTag,
        SecretOwner
    ]
    input_parameters += tokens

    environment = [
        PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
    ]

    def _create_startrek_ticket(self, summary, description):
        import startrek_client
        owner = utils.get_or_default(self.ctx, SecretOwner)
        secret = utils.get_or_default(self.ctx, StartrekSecretName)
        token = self.get_vault_data(owner, secret)
        st_client = startrek_client.Startrek(token=token, useragent='sandbox-task')
        return st_client.issues.create(
            queue=utils.get_or_default(self.ctx, StartrekQueue),
            summary=summary,
            description=description,
            tags=utils.get_or_default(self.ctx, TicketTag),
        )

    def _set_tokens(self, environment):
        owner = utils.get_or_default(self.ctx, SecretOwner)
        for token_class in self.tokens:
            secret = utils.get_or_default(self.ctx, token_class)
            if secret:
                token = self.get_vault_data(owner, secret)
                environment[token_class.ENV_VAR] = token

    def on_execute(self):
        juggler_ansible_resource = self.ctx.get('juggler_ansible_resource')
        if not juggler_ansible_resource or (juggler_ansible_resource == '0'):
            juggler_ansible_resource = channel.sandbox.list_releases(
                resource_type=resource_types.JUGGLER_ANSIBLE, limit=1
            )[0].resources[0].id
            self.ctx['juggler_ansible_resource'] = juggler_ansible_resource

        resource_path = self.sync_resource(juggler_ansible_resource)
        venv_path = self.path('juggler_ansible')
        tarfile.open(resource_path).extractall(venv_path)

        python_path = os.path.join(venv_path, 'bin', 'python')
        ansible_playbook_path = os.path.join(venv_path, 'bin', 'ansible-playbook')

        url = utils.get_or_default(self.ctx, PlaybookRepo)
        checkout_path = os.path.basename(url)
        git = Git(url)
        git.clone(checkout_path, 'master')

        environment = os.environ.copy()
        self._set_tokens(environment)

        files = utils.get_or_default(self.ctx, PlaybookPath)
        failed_playbook = []
        applied_playbook = 0

        # Find file into checkout_path but return path to file without checkout_path
        for filename in [f.replace(checkout_path + '/', '') for g in files
                         for f in glob.iglob(os.path.join(checkout_path, g))]:
            # Sometimes ansible-playbook fails. Try to apply playbook three times
            for attempt in xrange(0, 3):
                p = process.run_process(
                    [python_path, ansible_playbook_path, filename],
                    log_prefix=filename.replace('/', '_'), work_dir=checkout_path,
                    environment=environment, check=False)
                if p.returncode == 0:
                    applied_playbook += 1
                    break
                if attempt == 2:
                    failed_playbook.append(filename)

        self.set_info("Applied {} playbooks".format(applied_playbook))

        if failed_playbook:
            if utils.get_or_default(self.ctx, CreateTicket):
                summary = 'Failed to apply some playbook {}'.format(self.id)
                description = 'Task (({0} {0}))\n\nFailed playbook:\n{1}'.format(
                    self.http_url(), '\n- '.join(failed_playbook))
                issue = self._create_startrek_ticket(summary, description)
                self.set_info('https://st.yandex-team.ru/{}'.format(issue.key))

            raise errors.SandboxTaskFailureError('Failed to apply playbooks: ' + ', '.join(failed_playbook))


__Task__ = RunJugglerMarketPlaybook
