# coding: utf-8

import re

from sandbox.projects.common import utils
from sandbox.sandboxsdk import parameters
from sandbox.sandboxsdk import task
from sandbox.sandboxsdk.environments import PipEnvironment


class StartrekQueueFrom(parameters.SandboxStringParameter):
    name = 'startrek_queue_from'
    description = 'Startrek queue source'
    default_value = 'CSADMIN'


class StartrekQueueTo(parameters.SandboxStringParameter):
    name = 'startrek_queue_to'
    description = 'Startrek queue destination'
    default_value = 'TOOLSADMIN'


class HostPattern(parameters.SandboxStringParameter):
    name = 'host_pattern'
    description = 'Host pattern'
    default_value = 'cs-toolsfront|cs-minitools|conductor|logstorage|cs-mongotools|vs-mysqltools|startrek|cs-toolsback|vs-hotback|vs-otrs'


class SecretOwner(parameters.SandboxStringParameter):
    name = 'secret_owner'
    description = 'Vault secret owner'
    default_value = 'MARKETSRE'


class SecretName(parameters.SandboxStringParameter):
    name = 'secret_name'
    description = 'Vault secret name with Startrek token'
    default_value = 'startrek_csadmin_token'


class RunMarketFwMover(task.SandboxTask):
    """
    Movement of the matching tasks in another startrek queue
    """

    type = 'RUN_MARKET_FW_MOVER'

    execution_space = 100

    input_parameters = [
        StartrekQueueFrom,
        StartrekQueueTo,
        HostPattern,
        SecretOwner,
        SecretName,
    ]

    environment = [
        PipEnvironment('startrek_client', custom_parameters=['requests==2.18.4']),
    ]

    def on_execute(self):

        import startrek_client

        owner = utils.get_or_default(self.ctx, SecretOwner)
        secret = utils.get_or_default(self.ctx, SecretName)
        startrek_queue_from = utils.get_or_default(self.ctx, StartrekQueueFrom)
        startrek_queue_to = utils.get_or_default(self.ctx, StartrekQueueTo)
        host_pattern = utils.get_or_default(self.ctx, HostPattern)

        token = self.get_vault_data(owner, secret)
        st_client = startrek_client.Startrek(token=token, useragent='sandbox-task RUN_MARKET_FW_MOVER')

        query = 'Queue: {} and Summary: "golem fw:" and Resolution: empty()'.format(
            startrek_queue_from
        )
        issues = st_client.issues.find(query)
        keys_for_movement = []
        for issue in issues:
            re_result = re.search(host_pattern, issue.description)
            if re_result is not None:
                keys_for_movement.append(issue.key)

        if len(keys_for_movement) > 0:
            bulkchange = st_client.bulkchange.move(keys_for_movement, startrek_queue_to)
            bulkchange.wait()


__task__ = RunMarketFwMover
