from __future__ import absolute_import, unicode_literals

import logging

import os
import requests
from sandbox.projects.common import binary_task
from sandbox.projects.common.task_env import TinyRequirements

from sandbox import sdk2


class Solomon(requests.Session):
    BASE_URL = 'https://solomon.yandex-team.ru/api/v2'

    def __init__(self, token=None):
        super().__init__()
        if token is None:
            if 'SOLOMON_TOKEN' in os.environ:
                token = os.environ['SOLOMON_TOKEN']
            elif 'SOLOMON_TOKEN_PATH' in os.environ:
                with open(os.path.expanduser(os.environ['SOLOMON_TOKEN_PATH'])) as f:
                    token = f.read().strip()
            else:
                raise RuntimeError('Solomon token is required')
        self.headers.update({
            'Accept': 'application/json;charset=UTF-8',
            'Content-Type': 'application/json;charset=UTF-8',
            'Authorization': 'OAuth {}'.format(token)
        })

        self.projects = {}

    def get_project(self, project):
        if project not in self.projects:
            logging.info('Making request to {}/projects/{}'.format(self.BASE_URL, project))
            response = self.get('{}/projects/{}'.format(self.BASE_URL, project))
            if response.status_code == 404:
                self.projects[project] = None
            else:
                response.raise_for_status()
                self.projects[project] = response.json()

        return self.projects[project]


class MarketAlertsConfigsTestsError(Exception):
    pass


class MarketAlertsConfigsTests(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """
    Pre commit tests for market alerts configs
    """
    class Requirements(TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String('Arcadia url', default='arcadia:/arc/trunk/arcadia', required=True)
        malec_token = sdk2.parameters.YavSecret('Malec token', required=True)
        malec_config_root = sdk2.parameters.String('Malec root config dir', required=True)
        malec_config_globals = sdk2.parameters.String('Malec globals config dir', required=True)
        with sdk2.parameters.RadioGroup("Test type") as test_type:
            test_type.values["namespaces"] = test_type.Value("Test namespaces", default=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    def on_execute(self):
        malec_key = 'malec'
        if self.Parameters.malec_token.default_key:
            malec_key = self.Parameters.malec_token.default_key

        self.binary_on_execute(str(self.Parameters.arcadia_url), self.Parameters.malec_token.data()[malec_key],
                               str(self.Parameters.malec_config_root), str(self.Parameters.malec_config_globals),
                               str(self.Parameters.test_type))

    def binary_on_execute(self, arcadia_url, malec_token, malec_config_root, malec_config_globals, test_type):
        if test_type == "namespaces":
            self.test_namespaces(arcadia_url, malec_token, malec_config_root, malec_config_globals)

    def test_namespaces(self, arcadia_url, malec_token, malec_config_root, malec_config_globals):
        from sandbox.projects.common.arcadia.sdk import mount_arc_path
        from market.sre.tools.market_alerts_configurator.lib.juggler_config import JugglerConfig
        from market.sre.tools.market_alerts_configurator.lib.common import get_config_content, get_empty_args

        args = get_empty_args()

        errors = []

        with Solomon(token=malec_token) as solomon:
            with mount_arc_path(arcadia_url) as mp:
                logging.info("Conf path: {}".format(os.path.join(mp, malec_config_root)))
                logging.info("Globals path: {}".format(os.path.join(mp, malec_config_globals)))
                args.globals_dir = os.path.join(mp, malec_config_globals)
                for root, _, files in os.walk(os.path.join(mp, malec_config_root)):
                    for entry in files:
                        if (entry.endswith('.yml') or entry.endswith('.yaml')) and entry != 'a.yaml':
                            logging.info('Making juggler_config of: {}'.format(os.path.join(root, entry)))
                            juggler_config = JugglerConfig(
                                args,
                                get_config_content(args, os.path.join(root, entry)).get('juggler'),
                                create_api=False
                            )
                            juggler_config.make_checks()

                            for check in juggler_config.checks:
                                logging.info('Checking namespace {} for file {}'.format(check['namespace'], entry))
                                project = solomon.get_project(check['namespace'])
                                if project is None:
                                    errors.append('Bad namespace {} in file {}'.format(check['namespace'], entry))

        if errors:
            raise MarketAlertsConfigsTestsError('\n'.join(errors))
