# coding=utf-8

import logging
import json
import re
import requests

from sandbox import sdk2

from sandbox.projects.common import task_env
from sandbox.projects.common import binary_task


class MarketDisasterRulesMonitoring(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(task_env.BuildRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        disaster_tag = sdk2.parameters.String(
            "Тег в Juggler для поиска алертов и правил", required=True, default="market_disaster")
        incident_managers_duty = sdk2.parameters.List(
            "Список календарей дежурств инцидентных менеджеров", required=True, default=[])
        queue = sdk2.parameters.String("Очередь для тикетов", required=True, default="CSADMIN")
        ticket_tag = sdk2.parameters.String(
            "Тег для поиска тикетов", required=True, default="market_disaster_rules_monitoring")
        secret = sdk2.parameters.YavSecret(
            "Yav секрет с OAuth токенами", required=True)
        with sdk2.parameters.Group("Параметры для новых правил") as new_rule_params:
            default_project = sdk2.parameters.String(
                "Проект по умолчанию для всех новых правил", required=True, default="market.base_notifications")
            on_success_next_call_delay = sdk2.parameters.Integer(
                "Сколько секунд ждать после успешного дозвона в секундах",
                description="Допустимые значения от 60 до 1800",
                required=True, default=900)
            delay = sdk2.parameters.Integer(
                "Задержка перед первым звонком после перехода проверки в крит", required=True, default=900)
            time_start = sdk2.parameters.Integer(
                "Время начала эскалаций",
                description="Если событие случилось за пределами указанного диапазона - эскалация не начинается. "
                            "Допустимые значения от 0 до 23",
                required=True, default=11)
            time_end = sdk2.parameters.Integer(
                "Время конца эскалаций",
                description="Если событие случилось за пределами указанного диапазона - эскалация не начинается. "
                            "Допустимые значения от 0 до 23",
                required=True, default=19)
            day_start = sdk2.parameters.Integer(
                "Начальный день недели когда разрешено отправлять нотификации",
                description="1 - понедельник, 7 - воскресенье",
                required=True, default=1)
            day_end = sdk2.parameters.Integer(
                "Конечный день недели когда разрешено отправлять нотификации",
                description="1 - понедельник, 7 - воскресенье",
                required=True, default=5)
            repeat = sdk2.parameters.Integer(
                "Cколько раз звонить по указанному списку логинов",
                required=True, default=1)
            call_tries = sdk2.parameters.Integer(
                "Cколько раз пытаться дозвониться пользователю",
                description="Допустимые значения от 1 до 5",
                required=True, default=2)
            restart_after = sdk2.parameters.Integer(
                "Через сколько секунд надо рестартовать завершившуюся эскалацию, если она продолжает гореть в CRIT",
                description="Допустимые значения от 1800 до 604800",
                required=True, default=604800)
        ext_params = binary_task.binary_release_parameters(stable=True)
        with sdk2.parameters.Output:
            is_changed = sdk2.parameters.Bool("Правила уведомлений были изменены")

    def _get_token(self):
        yav_secret = self.Parameters.secret.data()
        return yav_secret["token"]

    def _get_tvm_secret(self):
        yav_secret = self.Parameters.secret.data()
        return yav_secret["tvm_secret"]

    def _get_tvm_id(self):
        yav_secret = self.Parameters.secret.data()
        return yav_secret["tvm_id"]

    def _get_abc_client(self, token):
        from market.sre.library.python.abc_client import AbcClient
        return AbcClient(token)

    def _get_juggler_api_client(self, token):
        from market.sre.library.python.juggler_api_client import JuggerApiClient
        return JuggerApiClient(token)

    def _get_nanny_client(self, token):
        from market.sre.library.python.nanny_client import NannyClient
        return NannyClient(token)

    def _get_yd_client(self, token):
        from market.sre.library.python.yd_client import YdClient
        return YdClient(token)

    def _get_startrek_client(self, token):
        from startrek_client import Startrek
        return Startrek(useragent="sb_task_market_disaster_rules_monitoring", token=token)

    def get_tvm_service_ticket(self, dst_tvm_id):
        from tvm2 import TVM2
        from tvm2.protocol import BlackboxClientId

        tvm = TVM2(
            client_id=self._get_tvm_id(),
            secret=self._get_tvm_secret(),
            blackbox_client=BlackboxClientId.Prod,
        )
        return tvm.get_service_ticket(dst_tvm_id)

    def _is_market_id(self, service_id):
        return service_id.startswith("production_") or service_id.startswith("prod_")

    def _get_abc_services_from_nanny(self, nanny_client, abc_client):
        abc_services = set()
        all_services = nanny_client.list_services_by_category("/market")
        services = filter(lambda service: self._is_market_id(service["_id"]), all_services)
        for service in services:
            abc_id = service["info_attrs"]["content"].get("abc_group")
            if abc_id and abc_id != 0:
                abc_service_info = abc_client.get_service_info_by_id(abc_id)
                abc_services.add(abc_service_info.get("slug"))
        logging.info("ABC services in Nanny: %s", ", ".join(abc_services))
        return abc_services

    def _validate_abc(self, abc_client, abc):
        abc = str(abc)
        if abc.isdigit():
            try:
                service_info = abc_client.get_service_info_by_id(abc)
                return service_info.get("slug")
            except requests.HTTPError:
                logging.warning("ABC сервис c ID \"%s\" не существует", abc)
        else:
            service_info = abc_client.get_service_info_by_slug(abc)
            if service_info.get("results"):
                return abc
        logging.warning("ABC сервис \"%s\" не существует", abc)
        return None

    def _get_abc_services_from_yd(self, yd_client, abc_client):
        abc_services = set()
        errors = set()
        all_stages = yd_client.get_all_stages()
        market_stages = [s for s in all_stages if re.match(r"^production_market_.+$", s)]
        for stage_id in market_stages:
            abcs = yd_client.get_abcs_for_stage(stage_id)
            for abc in abcs:
                if abc:
                    abc_slug = self._validate_abc(abc_client, abc)
                    if abc_slug:
                        abc_services.add(abc_slug)
                    else:
                        errors.add("ABC сервис \"{0}\" не существует. Он указан в "
                                   "((https://deploy.yandex-team.ru/stages/{1} {1})).".format(abc, stage_id))
        logging.info("ABC services in YD: %s", ", ".join(abc_services))
        return abc_services, sorted(errors)

    def _get_service_tag(self, abc_service_slug):
        return "market_{}_disaster".format(abc_service_slug.lower())

    def _get_disaster_notification_rules(self, juggler_client):
        notification_filter = juggler_client.get_notification_filter_by_tags([self.Parameters.disaster_tag])
        result = juggler_client.get_notify_rules_by_filter(notification_filter)
        return result.get("rules", [])

    def _get_map_abc_rules(self, abc_services, notification_rules):
        map_abc_rules = {}
        for abc_service in abc_services:
            map_abc_rules[abc_service] = []
            for rule in notification_rules.copy():
                if "tag={}".format(self._get_service_tag(abc_service)) in rule.get("selector", ""):
                    map_abc_rules[abc_service].append(rule)
                    notification_rules.remove(rule)
        obsolete_rules_id = [rule["rule_id"] for rule in notification_rules]
        logging.warning("Obsolete rules: %s", ", ".join(obsolete_rules_id))
        logging.info("Map ABC - Rules: %s", str(map_abc_rules))
        return map_abc_rules

    def _get_heads_of_abc(self, abc_client, abc_service_slug):
        return abc_client.get_members(service_slug=abc_service_slug, role_id=1)

    def _get_abc_schedule_slugs(self, abc_client, service_slug, tvm_ticket):
        duty_schedules = []
        duty_schedules.extend(abc_client.get_duty_schedules(service_slug=service_slug).get("results"))
        duty_schedules.extend(abc_client.get_duty_schedules_2_0(service_slug, tvm_ticket).get("result"))
        schedule_slugs = [x.get("slug") for x in duty_schedules if x.get("slug")]
        return schedule_slugs

    def _generate_notification_rule_phone(self, abc_client, abc_service_slug, abc_schedule_slugs):
        logins = ["@svc_{}:{}".format(abc_service_slug, x) for x in abc_schedule_slugs] + \
            self._get_heads_of_abc(abc_client, abc_service_slug) + \
            self.Parameters.incident_managers_duty
        if not logins:
            return None
        return {
            "project": self.Parameters.default_project,
            "selector": "tag={} & tag={}".format(self._get_service_tag(abc_service_slug), self.Parameters.disaster_tag),
            "template_name": "phone_escalation",
            "match_raw_events": False,
            "template_kwargs": {
                "on_success_next_call_delay": self.Parameters.on_success_next_call_delay,
                "delay": self.Parameters.delay,
                "time_start":  self.Parameters.time_start,
                "time_end":  self.Parameters.time_end,
                "day_start":  self.Parameters.day_start,
                "day_end":  self.Parameters.day_end,
                "repeat": self.Parameters.repeat,
                "call_tries": self.Parameters.call_tries,
                "restart_after":  self.Parameters.restart_after,
                "logins": logins
            },
            "check_kwargs": {},
            "description": "Base escalation for service {}".format(abc_service_slug)
        }

    def _check_rules_with_no_phone_escalation(self, abc_client, abc_tvm_ticket, map_abc_rules):
        rules = []
        errors = []
        for abc_service in map_abc_rules:
            for rule in map_abc_rules[abc_service]:
                if rule["template_name"] == "phone_escalation":
                    break
            else:
                logging.warning(
                    "For ABC ((https://abc.yandex-team.ru/services/{0}/ {0})) not found notification rules with "
                    '"template_name"="phone_escalation" and tag "market_{1}_disaster". It will be created.'
                        .format(abc_service, abc_service.lower())
                )
                schedule_slugs = self._get_abc_schedule_slugs(abc_client, abc_service, abc_tvm_ticket)
                rule = self._generate_notification_rule_phone(abc_client, abc_service, schedule_slugs)
                if rule:
                    rules.append(rule)
                else:
                    errors.append(
                        "Для ABC сервиса ((https://abc.yandex-team.ru/services/{0}/ {0})) нет правил уведомлений c "
                        '"template_name"="phone_escalation" и тегом "market_{1}_disaster".'
                        "Не удалось определить список логинов получателей."
                            .format(abc_service, abc_service.lower())
                    )
        return rules, errors

    def _prepare_rule_for_update(self, rule):
        for key in ("creation_time", "hints"):
            del rule[key]

    def _fix_rules_with_wrong_logins(self, abc_client, map_abc_rules):
        rules = []
        errors = []
        for abc_service in map_abc_rules:
            for rule in map_abc_rules[abc_service]:
                rule_updated = False
                if rule["template_name"] == "phone_escalation":
                    logins = rule["template_kwargs"]["logins"]
                    heads = self._get_heads_of_abc(abc_client, abc_service)
                    incident_managers_duty = self.Parameters.incident_managers_duty
                    if heads:
                        for head in heads:
                            if head in logins:
                                break
                        else:
                            logging.warning("Add %s to rule %s", ", ".join(heads), rule["rule_id"])
                            if incident_managers_duty and incident_managers_duty[0] in logins:
                                position = logins.index(incident_managers_duty[0])
                                for head in heads:
                                    logins.insert(position, head)
                            else:
                                logins.extend(heads)
                            rule_updated = True
                    else:
                        errors.append("В ABC сервисе ((https://abc.yandex-team.ru/services/{0}/ {0})) "
                            "нет руководителя.".format(abc_service))
                    for inc_manager_duty in incident_managers_duty:
                        if inc_manager_duty not in logins:
                            logging.warning("Add %s to rule %s", inc_manager_duty, rule["rule_id"])
                            logins.append(inc_manager_duty)
                            rule_updated = True
                if rule_updated:
                    self._prepare_rule_for_update(rule)
                    rules.append(rule)
        return rules, errors

    def _check_valid_duty_schedules(self, abc_client, abc_tvm_ticket, map_abc_rules):
        errors = []
        for abc_service in map_abc_rules:
            for rule in map_abc_rules[abc_service]:
                if rule["template_name"] == "phone_escalation":
                    shifts = [l for l in rule["template_kwargs"].get("logins") if l.startswith("@svc_")]
                    for shift in shifts:
                        match = re.match(r"^@svc_(?P<duty_abc>.+):(?P<duty_schedule>.+)$", shift)
                        service_slug = match.group("duty_abc")
                        schedule_slugs = self._get_abc_schedule_slugs(abc_client, service_slug, abc_tvm_ticket)
                        if match.group("duty_schedule") not in schedule_slugs:
                            errors.append(
                                "В правиле ((https://juggler.yandex-team.ru/notification_rules/?query=rule_id="
                                '{0} {0})) указано несуществующее дежурство "{1}".'.format(rule["rule_id"], shift)
                            )
        return errors

    def _fix_restart_after(self, map_abc_rules):
        rules = []
        for abc_service in map_abc_rules:
            for rule in map_abc_rules[abc_service]:
                if rule["template_name"] == "phone_escalation":
                    if "restart_after" not in rule["template_kwargs"]:
                        logging.warning("Fix restart_after to rule %s", rule["rule_id"])
                        rule["template_kwargs"]["restart_after"] = self.Parameters.restart_after
                        self._prepare_rule_for_update(rule)
                        rules.append(rule)
        return rules

    def _get_description(self, messages):
        if not messages:
            return "Все в порядке, теперь тикет можно закрыть."
        header = "Нашлись несколько отсутствующих или неправильных правил уведомлений:"
        footer = "\nТребования к правилам уведомлений и как правильно разбирать такой тикет описаны " \
                 "((https://wiki.yandex-team.ru/market/sre/fordev/" \
                 "%D0%9D%D0%B0%D1%81%D1%82%D1%80%D0%BE%D0%B9%D0%BA%D0%B0-%D1%8D%D1%81%D0%BA%D0%B0%D0%BB%D0%B0%D1%86" \
                 "%D0%B8%D0%B9-%D0%B4%D0%BB%D1%8F-%D0%B1%D0%B0%D0%B7%D0%BE%D0%B2%D1%8B%D1%85-%D0%BC%D0%BE%D0%BD%D0%B8" \
                 "%D1%82%D0%BE%D1%80%D0%B8%D0%BD%D0%B3%D0%BE%D0%B2-%D0%B2-%D1%81%D0%B5%D1%80%D0%B2%D0%B8%D1%81%D0%B0" \
                 "%D1%85-%D0%9C%D0%B0%D1%80%D0%BA%D0%B5%D1%82%D0%B0/ здесь))."
        formatted_messages = ["1. {}".format(m) for m in sorted(messages)]
        formatted_messages.insert(0, header)
        formatted_messages.append(footer)
        return "\n".join(formatted_messages)

    def _find_open_tickets(self, startrek_client, queue, tag):
        st_filter = "Queue: {} AND Status: !Closed AND Tags: {}".format(queue, tag)
        return startrek_client.issues.find(st_filter)

    def _create_ticket(self, startrek_client, queue, tag, description):
        return startrek_client.issues.create(
            queue=queue,
            summary="Обнаружены неправильные правила уведомлений с тегом market_disaster",
            description=description,
            tags=[tag]
        )

    def _update_tickets(self, tickets, description):
        for ticket in tickets:
            if ticket.description != description:
                ticket.update(description=description)
                logging.info("Ticket %s updated", ticket.key)
            else:
                logging.info("Ticket %s was not updated", ticket.key)

    def _create_or_update_ticket(self, startrek_client, error_messages):
        description = self._get_description(error_messages)
        logging.info(description)

        tickets = self._find_open_tickets(startrek_client, self.Parameters.queue, self.Parameters.ticket_tag)
        logging.info("Found tickets %s", ", ".join((t.key for t in tickets)))

        if error_messages and not tickets:
            new_ticket = self._create_ticket(
                startrek_client, self.Parameters.queue, self.Parameters.ticket_tag, description)
            logging.info("Created ticket %s", new_ticket.key)
            return

        if len(tickets) > 1:
            logging.warning("Too many open tickets: %s", ", ".join((t.key for t in tickets)))

        if tickets:
            self._update_tickets(tickets, description)

    def _update_rules(self, juggler_client, rules):
        errors = []
        for rule in rules:
            logging.info("Create or update rule %s", json.dumps(rule, indent=2))
            try:
                result = juggler_client.add_or_update_notify_rule(json.dumps(rule))
                logging.info("Rule id https://juggler.yandex-team.ru/notification_rules/?query=rule_id=%s",
                             result["result_id"])
                self.Parameters.is_changed = True
            except requests.HTTPError as err:
                if rule.get("rule_id"):
                    logging.error("Failed to update rule %s", rule["rule_id"])
                    errors.extend(
                        "Не удалось обновить правило ((https://juggler.yandex-team.ru/notification_rules/?query=rule_id="
                        "{0} {0})). <{Текст ошибки \n{1}}>".format(rule["rule_id"], err)
                    )
                else:
                    logging.error("Failed to create rule with selector %s", rule["selector"])
                    errors.extend("Не удалось создать правило с селектором {}. <{{Текст ошибки \n{}}}>".format(
                        rule["selector"], err))
        if not self.Parameters.is_changed:
            self.Parameters.is_changed = False
        return errors

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)

        logging.info("Starting task MARKET_DISASTER_RULES_MONITORING")

        oauth_token = self._get_token()
        nanny_client = self._get_nanny_client(oauth_token)
        yd_client = self._get_yd_client(oauth_token)
        abc_client = self._get_abc_client(oauth_token)
        abc_tvm_ticket = self.get_tvm_service_ticket(abc_client.WATCHER_PROD_TVM_ID)
        juggler_client = self._get_juggler_api_client(oauth_token)
        startrek_client = self._get_startrek_client(oauth_token)

        abc_services_from_nanny = self._get_abc_services_from_nanny(nanny_client, abc_client)
        abc_services_from_yd, yd_errors = self._get_abc_services_from_yd(yd_client, abc_client)
        abc_services = abc_services_from_nanny.union(abc_services_from_yd)

        disaster_notification_rules = self._get_disaster_notification_rules(juggler_client)
        map_abc_rules = self._get_map_abc_rules(abc_services, disaster_notification_rules)

        fixed_rules = []
        error_messages = []

        error_messages.extend(yd_errors)

        rules, errors = self._check_rules_with_no_phone_escalation(abc_client, abc_tvm_ticket, map_abc_rules)
        fixed_rules.extend(rules)
        error_messages.extend(errors)

        rules, errors = self._fix_rules_with_wrong_logins(abc_client, map_abc_rules)
        fixed_rules.extend(rules)
        error_messages.extend(errors)

        fixed_rules.extend(self._fix_restart_after(map_abc_rules))
        error_messages.extend(self._check_valid_duty_schedules(abc_client, abc_tvm_ticket, map_abc_rules))

        error_messages.extend(self._update_rules(juggler_client, fixed_rules))
        self._create_or_update_ticket(startrek_client, error_messages)
