import logging

from sandbox.projects.common import binary_task
from sandbox.projects.common.task_env import TinyRequirements

from sandbox import sdk2


class MarketSaltUpdateProductionError(Exception):
    pass


class MarketSaltUpdateProduction(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        arcadia_url = sdk2.parameters.String('Arcadia url', default='arcadia:/arc/trunk/arcadia', required=True)
        arc_token = sdk2.parameters.YavSecret('Arc token', required=True)
        s3_token = sdk2.parameters.YavSecret('S3 token', required=True)
        release_branch = sdk2.parameters.String('Salt release branch',
                                                default='releases/marketsre-salt', required=True)
        s3_bucket = sdk2.parameters.String('S3 bucket', required=True, default='market-configs')
        s3_path = sdk2.parameters.String('S3 bucket', required=True, default='salt/repo_version')

        ext_params = binary_task.binary_release_parameters(stable=True)

    def binary_on_execute(self, arc_token, s3_token, arcadia_url, release_branch, s3_bucket, s3_path):
        from sandbox.projects.common.vcs.arc import Arc
        from sandbox.projects.common.vcs.util import extract_path_details
        import boto3
        release_branch = release_branch.removesuffix('/production')

        _, revision, _ = extract_path_details(arcadia_url.strip())
        if revision is None:
            checkout = 'trunk'
        else:
            checkout = 'r{}'.format(revision)
        logging.info('Checkout target: {}'.format(checkout))

        arc = Arc(arc_oauth_token=arc_token)

        with arc.mount_path('', checkout, fetch_all=False) as mp:
            if revision is None:
                revision = arc.describe(mp, svn=True).strip().lstrip('r')
            logging.info('Revision: {}'.format(revision))

            if not arc.fetch(mp, '{}/v{}'.format(release_branch, revision)):
                logging.info('Creating release branch: {}'.format('{}/v{}'.format(release_branch, revision)))
                arc.checkout(mp, branch='production', create_branch=True)
                arc.push(mp, upstream='{}/v{}'.format(release_branch, revision))

            access_key_id, secret_access_key = s3_token.split(':', 1)

            logging.info('Setting S3 repo_version to: {}'.format(revision))
            s3 = boto3.resource('s3', aws_access_key_id=access_key_id,
                                aws_secret_access_key=secret_access_key,
                                endpoint_url='https://s3.mds.yandex.net')
            s3.Object(s3_bucket, s3_path).put(Body=str(revision), ACL='public-read')

    def on_execute(self):
        arc_key = 'arc'
        if self.Parameters.arc_token.default_key:
            arc_key = self.Parameters.arc_token.default_key

        s3_key = 's3'
        if self.Parameters.s3_token.default_key:
            s3_key = self.Parameters.s3_token.default_key

        self.binary_on_execute(self.Parameters.arc_token.data()[arc_key],
                               self.Parameters.s3_token.data()[s3_key],
                               str(self.Parameters.arcadia_url), str(self.Parameters.release_branch),
                               str(self.Parameters.s3_bucket), str(self.Parameters.s3_path))
