import logging
import hashlib
import os
import json

from sandbox.projects.common import binary_task
from sandbox.projects.common.task_env import TinyRequirements

from sandbox import sdk2

ARCANUM_REVIEW_BASE = "https://a.yandex-team.ru/review/"


class MarketSreKeysSync(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """MarketSreYPMigration Task"""

    class Requirements(TinyRequirements):
        disk_space = 1 * 1024

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        abc_service = sdk2.parameters.String("abc service", required=True, default="marketito")
        abc_role = sdk2.parameters.String("abc role", required=True, default="administration")
        arc_user = sdk2.parameters.String("arc user", required=True)
        arc_token = sdk2.parameters.YavSecret("arc token", required=True)
        abc_token = sdk2.parameters.YavSecret("abc token", required=True)
        output = sdk2.parameters.String("output", required=True, default="market/sre/conf/config_market_admin_keys")

        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Output:
            result_pr = sdk2.parameters.Integer("Arcadia PR")

    @staticmethod
    def _get_pr_num(out):
        if isinstance(out, bytes):
            out = out.decode("utf-8")
        if ARCANUM_REVIEW_BASE not in out:
            raise ValueError("invalid output from pr create: {}".format(out))
        pr_num = out.split(ARCANUM_REVIEW_BASE)[1].strip()
        return int(pr_num)

    def binary_on_execute(self, output, abc_service, abc_role, arc_user, arc_token, abc_token):
        from market.sre.conf.config_market_admin_keys.tool.lib import get_skotty_secure_pubs, get_keys
        from sandbox.projects.abc.client import AbcClient
        from sandbox.projects.common.vcs.arc import Arc

        abc_client = AbcClient(abc_token)

        service_info = abc_client.get_service_info_by_slug(abc_service.lower())

        members = abc_client.get_people_from_service(
            service_id=service_info["id"],
            role_scope=abc_role.lower(),
            is_robot=False
        )
        logging.info("User list: {}".format(members))
        members = set(members)

        s = hashlib.md5("\n".join(get_keys(sorted(list(members)), secure_pubs=get_skotty_secure_pubs())).encode("utf-8")).hexdigest()

        arc = Arc(arc_oauth_token=arc_token)

        with arc.mount_path("", "trunk", fetch_all=False) as mp:
            arc.checkout(mp, branch="market-sre-keys-sync", create_branch=True)

            base_pw = os.path.join(mp, output)

            if not os.path.exists(base_pw):
                os.mkdir(base_pw)

            with open(os.path.join(base_pw, "md5.sum"), "w") as f:
                f.write(s)

            with open(os.path.join(base_pw, "user_list.json"), "w") as f:
                json.dump(sorted(list(members)), f, indent=2)

            status = arc.status(mp, True)

            logging.info(status)
            if "changed" in status["status"] and len(status["status"]["changed"]) > 0:
                logging.info("Committing changes")

                arc.add(mp, os.path.join(output, "md5.sum"))
                arc.add(mp, os.path.join(output, "user_list.json"))

                commit_message = "Automatic: update market sre keys list"
                arc.commit(mp, commit_message)

                arc.push(mp, upstream="users/{}/{}".format(arc_user, "market-sre-keys-sync"), force=True)

                pr_num = self._get_pr_num(
                    arc.pr_create(mp, publish=False, auto=True, message=commit_message, no_commits=True))
                self.Parameters.result_pr = pr_num
                self.set_info(
                    'Arcanum <a href="{}{}">PR:{}</a>.'.format(ARCANUM_REVIEW_BASE, pr_num, pr_num),
                    do_escape=False)

    def on_execute(self):
        arc_key = 'arc'
        if self.Parameters.arc_token.default_key:
            arc_key = self.Parameters.arc_token.default_key

        abc_key = 'abc'
        if self.Parameters.abc_token.default_key:
            abc_key = self.Parameters.abc_token.default_key

        return self.binary_on_execute(str(self.Parameters.output),
                                      str(self.Parameters.abc_service),
                                      str(self.Parameters.abc_role),
                                      str(self.Parameters.arc_user),
                                      self.Parameters.arc_token.data()[arc_key],
                                      self.Parameters.abc_token.data()[abc_key])
