from datetime import datetime
import re
import logging

from sandbox.projects.common import binary_task
from sandbox.projects.common.task_env import TinyRequirements

from sandbox import sdk2

DB_SCHEMA = [
    {"name": "datetime", "type": "uint64", "sort_order": "ascending", "required": True},
    {"name": "dc", "type": "string", "required": True},
    {"name": "yp_services", "type": "float", "required": True},
    {"name": "yp_instances", "type": "float", "required": True},
    {"name": "yp_cpu", "type": "float", "required": True},
    {"name": "yp_memory", "type": "float", "required": True},
    {"name": "yp_hdd", "type": "float", "required": True},
    {"name": "yp_ssd", "type": "float", "required": True},
    {"name": "gencfg_services", "type": "float", "required": True},
    {"name": "gencfg_instances", "type": "float", "required": True},
    {"name": "gencfg_cpu", "type": "float", "required": True},
    {"name": "gencfg_memory", "type": "float", "required": True},
    {"name": "gencfg_hdd", "type": "float", "required": True},
    {"name": "gencfg_ssd", "type": "float", "required": True},
]

DB_SIMPLE_SCHEMA = [
    {"name": "datetime", "type": "uint64", "sort_order": "ascending", "required": True},
    {"name": "yp_services", "type": "uint64", "required": True},
    {"name": "gencfg_services", "type": "uint64", "required": True},
]


class MarketSreYPMigration(binary_task.LastBinaryTaskRelease, sdk2.Task):
    """MarketSreYPMigration Task"""

    class Requirements(TinyRequirements):
        pass

    class Parameters(sdk2.Task.Parameters):
        # common parameters
        kill_timeout = 3600

        # custom parameters
        yt_cluster = sdk2.parameters.String("YT cluster", default="hahn", required=True)
        source_path = sdk2.parameters.String("Source path", required=True)
        output_path = sdk2.parameters.String("Output path", required=True)
        yt_token = sdk2.parameters.YavSecret("yt token", required=True)
        nanny_token = sdk2.parameters.YavSecret("nanny token", required=True)

        ext_params = binary_task.binary_release_parameters(stable=True)

    @staticmethod
    def get_db_path(db_path):
        return str(db_path).rstrip("/")

    @staticmethod
    def timestamp(dt):
        return (dt - datetime(1970, 1, 1)).total_seconds()

    def read_one_table(self, dt, table, yt_client):
        resources = {"services", "instances", "cpu", "memory", "hdd", "ssd"}
        dcs = {"SAS", "VLA", "MAN", "IVA", "MSK", "ALL", "MULTI", "NODC"}
        yp_quota = {d: {r: 0 for r in resources} for d in dcs}
        gencfg_quota = {d: {r: 0 for r in resources} for d in dcs}

        def add_quota(row, storage):
            dc = row["dc"]
            if "," in dc:
                dc = "MULTI"
            elif dc == "":
                dc = "NODC"
            storage[dc.upper()]["services"] += 1
            storage["ALL"]["services"] += 1
            for r in resources - {"services"}:
                storage[dc.upper()][r] += row[r]
                storage["ALL"][r] += row[r]

        for table_row in yt_client.read_table(table):
            if table_row["is_yp"]:
                add_quota(table_row, yp_quota)
            else:
                add_quota(table_row, gencfg_quota)

        dt = dt.replace(hour=0, minute=0, second=0, microsecond=0)

        return [dict(
            datetime=int(self.timestamp(dt)),
            dc=d,
            **{"{}_{}".format(t, r): float(q[d][r])
               for r in resources
               for t, q in {"yp": yp_quota, "gencfg": gencfg_quota}.items()}
        ) for d in dcs]

    def graf_from_tables(self, yt_client, source_path, output_path):
        import yt.wrapper as yt
        logging.info("Generate report from tables")
        source_yt_path = yt.YPath(self.get_db_path(source_path))
        output_yt_path = yt.YPath(self.get_db_path(output_path))
        output_table = yt.TablePath(output_yt_path.join("graf"), schema=DB_SCHEMA)
        result = []
        logging.info("table: services")
        result.extend(
            self.read_one_table(datetime.now(), yt.TablePath(source_yt_path.join("services")), yt_client)
        )
        for table in yt_client.list(source_yt_path.join("history")):
            if datetime.now().strftime("%Y-%m-%d") == table:
                continue
            logging.info("table: history/{}".format(table))
            dt = datetime.strptime(table, "%Y-%m-%d")
            result.extend(
                self.read_one_table(dt, yt.TablePath(source_yt_path.join("history").join(table)), yt_client)
            )

        result = sorted(result, key=lambda i: i['datetime'])

        yt_client.write_table(output_table, result)

    def graf_from_nanny_api(self, yt_client, manager, output_path):
        import yt.wrapper as yt
        logging.info("Generate report from nanny")
        services = manager.list_services(
            login="",
            fn_filter=lambda x: re.search(r"^mt_", x.service_id) is None,
            fn_category=lambda x: re.search(r"^/market/", x.category)
        )
        yp_services = 0
        gencfg_services = 0

        for service_info in services:
            service_id = service_info.service_id
            try:
                logging.info("nanny service: {}".format(service_id))
                runtime_attrs = manager.get_runtime_attrs(service_id)
                if runtime_attrs.instances.chosen_type == "EXTENDED_GENCFG_GROUPS":
                    gencfg_services += 1
                elif runtime_attrs.instances.chosen_type == "YP_POD_IDS":
                    yp_services += 1
            except Exception as exc:
                raise RuntimeError('Error for service: {}, {}: {}'.format(service_id, exc.__class__.__name__, exc))

        output_yt_path = yt.YPath(self.get_db_path(output_path))
        output_table = yt.TablePath(output_yt_path.join("graf_simple"), schema=DB_SIMPLE_SCHEMA)

        dt = datetime.now()
        dt = dt.replace(hour=0, minute=0, second=0, microsecond=0)

        if not yt_client.exists(output_table):
            yt_client.create('table', output_table, ignore_existing=True, attributes={
                'dynamic': True,
                'schema': DB_SIMPLE_SCHEMA,
                'enable_dynamic_store_read': True
            })
            from time import sleep
            sleep(15)
            yt_client.mount_table(output_table)
            sleep(15)

        yt_client.insert_rows(output_table, [
            {"datetime": int(self.timestamp(dt)),
             "yp_services": int(yp_services),
             "gencfg_services": int(gencfg_services)}
        ])

    def binary_on_execute(self, yt_cluster, source_path, output_path, yt_token, nanny_token):
        import yt.wrapper as yt
        from market.sre.tools.rtc import nanny
        yt_client = yt.YtClient(proxy=yt_cluster, token=yt_token)
        manager = nanny.create_manager(nanny_token=nanny_token, with_vault_client=True)

        self.graf_from_tables(yt_client, source_path, output_path)
        self.graf_from_nanny_api(yt_client, manager, output_path)

    def on_execute(self):
        key = 'yt'
        if self.Parameters.yt_token.default_key:
            key = self.Parameters.yt_token.default_key

        key_nanny = 'nanny'
        if self.Parameters.nanny_token.default_key:
            key_nanny = self.Parameters.nanny_token.default_key

        return self.binary_on_execute(self.Parameters.yt_cluster, self.Parameters.source_path,
                                      self.Parameters.output_path, self.Parameters.yt_token.data()[key],
                                      self.Parameters.nanny_token.data()[key_nanny])
