# coding=utf-8

import codecs
import logging
import json

from sandbox import sdk2

from sandbox.projects.market.stock_storage import StockStorageAmmo, StockStorageShootingConfig
from sandbox.projects.market.stock_storage.ammo_generator import get_ammo_string
from sandbox.projects.market.stock_storage.freezeStocksGenerator import FreezeStocksGenerator
from sandbox.projects.market.stock_storage.pushStocksGenerator import PushStocksGenerator


class HumanLikeGenerator(sdk2.Task):
    class Parameters(sdk2.Task.Parameters):
        logging.info('Start init data')
        ss_url = sdk2.parameters.String('SS Url', required=True, default='https://bos.tst.vs.market.yandex.net')
        stocks_in_order = sdk2.parameters.Integer('Amount of different stocks for freeze in ammo', required=True,
                                                  default=5)
        stocks_in_push = sdk2.parameters.Integer('Amount of different stocks for pushStocks in ammo', required=True,
                                                 default=25)
        vendor_ids = sdk2.parameters.List('IDs of vendors', required=False, default=[2])
        warehouse_ids = sdk2.parameters.List('IDs of warehouse', required=False, default=[1])
        orders_amount = sdk2.parameters.Integer('Amount of orders', required=False, default=10000)
        get_available_by_order_ratio = sdk2.parameters.Integer('Ratio of getAvailable by order', required=False,
                                                               default=30)
        push_stocks_by_order_ratio = sdk2.parameters.Float('Ratio of pushStocks by order', required=False, default=0.05)
        shooting_length_minutes = sdk2.parameters.Integer('Shooting length in minutes', required=False, default=20)
        default_rps_config = sdk2.parameters.String('Default RPS config', required=False, default='40')
        method_rps_overrides = sdk2.parameters.String('JSON in format', required=False,
                                                      default='{"getAvailableAmounts": 70, "freeze": 5, "pushStocks": 2}')
        logging.info('End init data')

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5

    class Constants:
        def __init__(self):
            pass

        SHOOTING_CONFIG_TEMPLATE = 'uploader:\n' \
                                   '  enabled: true\n' \
                                   '  task: MARKETFF-3289\n' \
                                   '  api_address: https://lunapark.yandex-team.ru/\n' \
                                   '  job_name: MARKETFF-3289\n' \
                                   '  operator: tishins\n' \
                                   '  package: yandextank.plugins.DataUploader\n\n' \
                                   '' \
                                   'neuploader:\n' \
                                   '  enabled: false\n' \
                                   '  package: yandextank.plugins.NeUploader\n\n' \
                                   '' \
                                   'autostop:\n' \
                                   '  enabled: true\n' \
                                   '  autostop: [\'time(25000ms,2m)\', \'http(5xx,10%%,2m)\']\n' \
                                   '  package: yandextank.plugins.Autostop\n' \
                                   '  report_file: autostop_report.txt\n\n' \
                                   '' \
                                   'phantom:\n' \
                                   '  enabled: true\n' \
                                   '  package: yandextank.plugins.Phantom\n' \
                                   '  timeout: 30s\n' \
                                   '  load_profile: {load_type: rps, schedule: \'const(%s,%dm)\'}\n'
        MULTI_SECTION_TEMPLATE = '  multi:\n%s'
        MULTI_ELEMENT_TEMPLATE = '    - chosen_cases: %s\n' \
                                 '      load_profile: {load_type: rps, schedule: \'const(%s,%dm)\'}\n'

    def on_execute(self):
        logging.info(
            '''Main params:
            ss_url = {}
            stocks_in_order = {}
            stocks_in_push = {}
            vendor_ids = {}
            warehouse_ids = {}
            orders_amount = {}
            get_available_by_order_ratio = {}
            push_stocks_by_order_ratio = {}
            shooting_length_minutes = {}
            default_rps_config = {}
            '''.format(self.Parameters.ss_url, self.Parameters.stocks_in_order, self.Parameters.stocks_in_push,
                       self.Parameters.vendor_ids, self.Parameters.warehouse_ids, self.Parameters.orders_amount,
                       self.Parameters.get_available_by_order_ratio, self.Parameters.push_stocks_by_order_ratio,
                       self.Parameters.shooting_length_minutes, self.Parameters.default_rps_config,
                       self.Parameters.method_rps_overrides)
        )

        logging.info('Start ammo generating')
        stock_storage_ammo = self.create_freeze_stocks_generator().get_available_and_freeze_generator()
        logging.info('Freezes was generated')
        stock_storage_ammo.extend(self.create_push_stocks_generator().generate_push_stocks())
        logging.info('PushStocks was generated')
        ammo_string = get_ammo_string(stock_storage_ammo)
        logging.info('Start to write ammo to file')

        ammo = sdk2.ResourceData(StockStorageAmmo(self, 'push stocks', 'stock_storage_ammo.txt'))
        with codecs.open('stock_storage_ammo.txt', 'w', encoding='utf-8') as f:
            f.write(ammo_string)
        ammo.ready()

        print('Start to generate config')
        existing_tags = ['getAvailableAmounts', 'freeze', 'pushStocks']
        self.shooting_config_generator(existing_tags)

    def create_push_stocks_generator(self):
        stocks_generator = PushStocksGenerator()
        stocks_generator.Parameters.ss_url = self.Parameters.ss_url
        stocks_generator.Parameters.stocks_in_ammo = self.Parameters.stocks_in_push
        stocks_generator.Parameters.vendor_ids = self.Parameters.vendor_ids
        stocks_generator.Parameters.warehouse_ids = self.Parameters.warehouse_ids
        stocks_generator.Parameters.stocks_amount = int(
            self.Parameters.orders_amount * self.Parameters.push_stocks_by_order_ratio) + 1
        return stocks_generator

    def create_freeze_stocks_generator(self):
        freeze_stocks_generator = FreezeStocksGenerator()
        freeze_stocks_generator.Parameters.ss_url = self.Parameters.ss_url
        freeze_stocks_generator.Parameters.stocks_in_ammo = self.Parameters.stocks_in_order
        freeze_stocks_generator.Parameters.vendor_ids = self.Parameters.vendor_ids
        freeze_stocks_generator.Parameters.warehouse_ids = self.Parameters.warehouse_ids
        freeze_stocks_generator.Parameters.orders_amount = self.Parameters.orders_amount
        freeze_stocks_generator.Parameters.get_available_by_order_ratio = self.Parameters.get_available_by_order_ratio
        return freeze_stocks_generator

    def shooting_config_generator(self, existing_tags):
        shooting_config = self.Constants.SHOOTING_CONFIG_TEMPLATE % \
                          (self.Parameters.default_rps_config, self.Parameters.shooting_length_minutes)

        rps_overrides = json.loads(str(self.Parameters.method_rps_overrides))
        multi_elements = ''
        for key in existing_tags:
            key_rps_config = self.Parameters.default_rps_config
            logging.info('key_rps_config = {} for key = {}'.format(key_rps_config, key))
            if key in rps_overrides:
                key_rps_config = rps_overrides[key]
            multi_elements += self.Constants.MULTI_ELEMENT_TEMPLATE % \
                              (key, key_rps_config, self.Parameters.shooting_length_minutes)

        if len(multi_elements) > 0:
            multi_section = self.Constants.MULTI_SECTION_TEMPLATE % multi_elements
            shooting_config += multi_section

        config_data = sdk2.ResourceData(StockStorageShootingConfig(self, 'SS config', 'stock_storage_config.txt'))
        with codecs.open('stock_storage_config.txt', 'w', encoding='utf-8') as f:
            f.write(shooting_config)
        config_data.ready()
