# coding=utf-8

import logging
import requests
from sandbox import sdk2
from sandbox.sandboxsdk import environments


class UnmappedSkuDisabler(sdk2.Task):

    class Parameters(sdk2.Task.Parameters):
        ss_database_user_vault_key = sdk2.parameters.String('SS databse user vault key', required=True)
        ss_database_password_vault_key = sdk2.parameters.String('SS databse password vault key',
                                                                required=True)
        query_limit = sdk2.parameters.Integer('Limit for skus query', required=False)
        batch_size = sdk2.parameters.Integer('Proccessing batch size', required=True)

        yql_token_name = sdk2.parameters.String(
            "YQL token name",
            default='YQL_TOKEN',
            required=True
        )
        warehouse_id = sdk2.parameters.String('Warehouse id', required=True)
        partner_warehouse_ids = sdk2.parameters.String('Comma separated warehouse ids', required=True)
        ticket_key = sdk2.parameters.String('Tracker ticket key', required=True)
        tvm_service_ticket_vault_key = sdk2.parameters.String('TVM service ticket vault key', required=True)

    class Requirements(sdk2.Requirements):
        disk_space = 1024 * 5
        environments = (environments.PipEnvironment('psycopg2-binary'),
                        environments.PipEnvironment('yandex-yt'),
                        environments.PipEnvironment('yql'))

    class Sku:
        def __init__(self, sku, vendor_id, warehouse_id):
            self.sku = sku
            self.vendor_id = vendor_id
            self.warehouse_id = warehouse_id

    def get_crossdock_divergence_skus(self):

        conn = None
        cur = None
        try:
            logging.info('Fetching credentials')
            user = sdk2.Vault.data(self.Parameters.ss_database_user_vault_key)
            password = sdk2.Vault.data(self.Parameters.ss_database_password_vault_key)
            logging.info('Connecting to the SS database...')
            import psycopg2
            conn = psycopg2.connect(
                host="vla-ztlnd24dqcjd0m75.db.yandex.net",
                port="6432",
                database="market_stockdb",
                user=user,
                password=password)
            logging.info('Connected')

            cur = conn.cursor()
            logging.info('Getting crossdock divergence skus...')

            query = ("select cd.sku, cd.vendor_id, cd.warehouse_id "
                     "from sku cd "
                     "    left join sku wh on cd.sku = wh.sku "
                     "    and wh.vendor_id = cd.vendor_id "
                     "    and wh.warehouse_id = " + self.Parameters.warehouse_id +
                     "where cd.warehouse_id in (" + self.Parameters.partner_warehouse_ids + ") "
                     "  and wh.sku is null "
                     "  and cd.enabled = true "
                     "  and cd.created < now() - interval '4 hour' "
                     "order by cd.updated desc ")

            if self.Parameters.query_limit:
                query = query + "limit " + str(self.Parameters.query_limit)

            cur.execute(query)

            raw_result = cur.fetchall()
            result = []
            for row in raw_result:
                result.append(self.Sku(row[0], row[1], row[2]))

            logging.info("Found skus: " + str(len(result)))

            return result
        except Exception as error:
            logging.info(error)

        finally:
            if cur is not None:
                cur.close()
            if conn is not None:
                conn.close()
            logging.info('SS database connection closed.')

    def get_unmapped_skus(self, skus):

        yql_token = sdk2.Vault.data(self.Parameters.yql_token_name)

        from yt import wrapper
        client = wrapper.YtClient(proxy='hahn', token=yql_token)

        from yt import clickhouse as chyt

        conditions = []
        for sku in skus:
            conditions.append("(supplier_id = " + str(sku.vendor_id) + " and shop_sku = '" + sku.sku.decode("UTF-8") + "')")

        yql_query = ("SELECT shop_sku, supplier_id "
                     "FROM `//home/market/production/mbo/stat/mboc_offers_expanded_sku/actual-state` "
                     "WHERE (approved_market_sku_id is not null and availability == 'ACTIVE') and (" + " or ".join(conditions) + ")")

        mapped = list(chyt.execute(yql_query, alias='*ch_public', client=client))
        logging.info("Found mapped skus: " + str(len(mapped)))

        mapped_dict = {}
        for row in mapped:
            mapped_dict[row['shop_sku'] + '_' + str(row['supplier_id'])] = row

        result = []
        for sku in skus:
            if (sku.sku.decode('UTF-8') + "_" + str(sku.vendor_id)) not in mapped_dict:
                result.append(sku)

        return result

    def disable_skus(self, skus):
        logging.info("Disabling skus...")

        result=[]
        for sku in skus:
            result.append('{"shopSku":"' + str(sku.sku) + '","vendorId":"' + str(sku.vendor_id) + '","warehouseId":"' + str(sku.warehouse_id) + '"}')

        logging.info("Disabling:" + ", ".join(result))

        tvm_service_ticket = sdk2.Vault.data(self.Parameters.tvm_service_ticket_vault_key)

        logging.info(requests.post('https://bos.vs.market.yandex.net/support/stock-enable',
                                   data='{"enabled": false, "comment": "' + self.Parameters.ticket_key + '", "stocks": ['+", ".join(result)+']}',
                                   headers={'Content-Type': 'application/json', 'X-Ya-Service-Ticket': '' + tvm_service_ticket + ''}))

        return result

    def on_execute(self):
        skus = self.get_crossdock_divergence_skus()
        n = self.Parameters.batch_size
        for i in range(0, len(skus), n):
            chunk = skus[i:i + n]
            logging.info("Proccess skus from " + str(i) + " to " + str(i + n))
            unmapped_skus = self.get_unmapped_skus(chunk)
            self.disable_skus(unmapped_skus)
