from collections import defaultdict
import random

user_page_ids = {
    'GET_api_user',
    'GET_api_route-points',
    'GET_api_app_versions_latest',
    'GET_api_tasks',
    'GET_api_call-tasks',
    'GET_api_tasks_order-delivery',
    'GET_api_shifts_current',
    'GET_api_scanner_v2',
    'GET_api_help_promo',
    'GET_api_locker_info',
    'GET_api_locker_boxbot_postamat_open-cells',
    'GET_api_help',
    'GET_api_messages',
    'GET_api_help_promo_promoId',
}
page_id_params = {
    'GET_api_shifts_id_statistics_pay': {
        'group': 'user_shift',
        'id_position': -3
    },
    'GET_api_shifts_id_statistics': {
        'group': 'user_shift',
        'id_position': -2
    },
    'GET_api_shifts_id_params': {
        'group': 'user_shift',
        'id_position': -2
    },
    'GET_api_shifts_id_rescheduleDates': {
        'group': 'user_shift',
        'id_position': -2
    },
    'GET_api_route-points_id': {
        'group': 'route_point',
        'id_position': -1
    },
    'GET_api_tasks_order-delivery-single_task-id': {
        'group': 'task',
        'id_position': -1
    },
    'GET_api_call-tasks_id': {
        'group': 'task',
        'id_position': -1
    },
    'GET_api_tasks_locker-delivery_task-id': {
        'group': 'task',
        'id_position': -1
    },
    'GET_api_locker_boxbot_postamat_references_taskId': {
        'group': 'task',
        'id_position': -1
    },
}


def group_by_type(raw_definitions):
    groups = defaultdict(list)
    for d in raw_definitions:
        if 'entity_type' in d:
            group = d['entity_type']
            groups[group].append(d)
    return groups


def extract_entity_id_from_endpoint(endpoint, endpoint_page_id_params):
    id_position = endpoint_page_id_params['id_position']
    return int(endpoint.split('/')[id_position])


def fill_entity_type(raw_definition):
    page_id = raw_definition['page_id']
    if page_id in page_id_params.keys():
        endpoint_page_id_params = page_id_params[page_id]
        entity_type = endpoint_page_id_params['group']
        raw_definition['entity_type'] = entity_type
        endpoint = raw_definition['endpoint']
        entity_id = extract_entity_id_from_endpoint(endpoint, endpoint_page_id_params)
        raw_definition['entity_id'] = entity_id
    if page_id in user_page_ids:
        raw_definition['entity_type'] = 'user'


class RequestDefinitionService:
    def __init__(self, tpl_db):
        self.tpl_db = tpl_db

    def fill_request_definitions(self, raw_definitions):
        for d in raw_definitions:
            fill_entity_type(d)
        raw_definition_groups = group_by_type(raw_definitions)

        self.fill_user_params_for_user_shift_definitions(raw_definition_groups['user_shift'])
        self.fill_user_params_for_route_point_definitions(raw_definition_groups['route_point'])
        self.fill_user_params_for_task_definitions(raw_definition_groups['task'])
        user_emails = self.extract_user_emails(raw_definitions)
        self.fill_user_params_for_user_definitions(raw_definition_groups['user'], user_emails)
        return raw_definitions

    def fill_user_params_for_user_shift_definitions(self, user_shift_raw_definitions):
        if len(user_shift_raw_definitions) == 0:
            return
        user_shift_ids = [d['entity_id'] for d in user_shift_raw_definitions]
        user_mappings = self.tpl_db.get_user_shifts_user_mappings(user_shift_ids)
        for d in user_shift_raw_definitions:
            user_shift_id = d['entity_id']
            if user_shift_id in user_mappings:
                d['user_email'] = user_mappings[user_shift_id]['user_email']

    def fill_user_params_for_route_point_definitions(self, route_point_raw_definitions):
        if len(route_point_raw_definitions) == 0:
            return
        route_point_ids = [d['entity_id'] for d in route_point_raw_definitions]
        user_mappings = self.tpl_db.get_route_point_user_mappings(route_point_ids)
        for d in route_point_raw_definitions:
            route_point_id = d['entity_id']
            if route_point_id in user_mappings:
                d['user_email'] = user_mappings[route_point_id]['user_email']

    def fill_user_params_for_task_definitions(self, task_raw_definitions):
        if len(task_raw_definitions) == 0:
            return
        task_ids = [d['entity_id'] for d in task_raw_definitions]
        user_mappings = self.tpl_db.get_task_user_mappings(task_ids)
        for d in task_raw_definitions:
            task_id = d['entity_id']
            if task_id in user_mappings:
                d['user_email'] = user_mappings[task_id]['user_email']

    def extract_user_emails(self, raw_definitions):
        return [d['user_email'] for d in raw_definitions if 'user_email' in d]

    def fill_user_params_for_user_definitions(self, user_raw_definitions, user_emails):
        email = random.choice(user_emails)
        for d in user_raw_definitions:
            d['user_email'] = email
