# -*- coding: utf-8 -*-

import os
import io
import random
import logging
import json

from os.path import join as join_path, realpath, dirname

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sandboxsdk import environments
from sandbox.sandboxsdk.process import run_process
from datetime import date, timedelta


class MarketUgcBuildCpcInterviews(sdk2.Task):
    """
    Вычисляет необходимые поля (даты), выполняет yql-запрос, дожидается завершения, валидирует размер конечной таблицы.
    """

    dns = ctm.DnsType.DNS64
    client_tags = ctc.Tag.LINUX_TRUSTY
    YQL_READ_RESULT_TIMEOUT = 60 * 120  # sec

    class Parameters(sdk2.task.Parameters):
        target_date = sdk2.parameters.String(
            'Таблица с результатом будет привязана к указанному дню.  По умолчанию: "сегодня".', required=False, default_value='')

        env_type = sdk2.parameters.String(
            'Тип среды. Влияет на путь к таблицам. По умолчанию: "production"', required=False, default_value='')

    def on_execute(self):
        with environments.VirtualEnvironment() as venv:
            self._prepare_venv(venv)

            token_file = 'yql-token-%s' % random.randint(0, 1e6)
            self._write_to_file(token_file, self._get_yql_token())

            self._create_target_table_if_not_exists(token_file, venv)
            self._build_interviews(token_file, venv)
            self._check_poll_request_result(token_file, venv)
            self._check_order_delivery_result(token_file, venv)

    def _create_target_table_if_not_exists(self, token_file, venv):
        query_tpl_file_name = "createTableIfNotExists.tpl.sql"
        out_path = self._run_yql(token_file, query_tpl_file_name, venv)
        self._print_file(out_path)

    def _build_interviews(self, token_file, venv):
        query_tpl_file_name = "interviewsBuilder.tpl.sql"
        out_path = self._run_yql(token_file, query_tpl_file_name, venv)
        self._print_file(out_path)

    def _check_poll_request_result(self, token_file, venv):
        query_tpl_file_name = "checkPollRequestsResult.tpl.sql"
        out_path = self._run_yql(token_file, query_tpl_file_name, venv)
        self._print_file(out_path)
        if not self._result_data_is_correct(out_path):
            raise RuntimeError("Упала валидация таблицы poll_request.")

    def _check_order_delivery_result(self, token_file, venv):
        query_tpl_file_name = "checkOrderDeliveryResult.tpl.sql"
        out_path = self._run_yql(token_file, query_tpl_file_name, venv)
        self._print_file(out_path)
        if not self._result_data_is_correct(out_path):
            raise RuntimeError("Упала валидация таблицы order_delivery.")

    def _run_yql(self, token_file, yql_tpl_path, venv):
        request_path = join_path(dirname(realpath(__file__)), yql_tpl_path)
        yql_tpl = open(request_path).read().decode("utf-8")

        yql_body = self._fill_yql_template(yql_tpl)
        query_file = 'query-%s.sql' % random.randint(0, 1e6)
        self._write_to_file(query_file, yql_body)

        env = dict(os.environ.copy(), YQL_API_TIMEOUT=str(self.YQL_READ_RESULT_TIMEOUT))

        yql_cli_wrapper_path = join_path(dirname(realpath(__file__)), 'yqlCliWrapper.py')
        out_path = "{}_out".format(yql_tpl_path)
        run_process([
            venv.executable,
            yql_cli_wrapper_path,
            '--token-file={}'.format(token_file),
            '--query-file="{}"'.format(query_file),
            '--out={}'.format(out_path),
        ], log_prefix='request_yql', shell=True, environment=env, check=True)
        return out_path

    @staticmethod
    def _result_data_is_correct(filepath):
        if os.path.exists(filepath):
            js_content = json.loads(open(filepath).read().decode("utf-8"))
            return int(js_content[0]["totalRowCount"]) > 25000
        else:
            raise RuntimeError("Валидация не прошла - файл {} не существует.".format(filepath))

    @staticmethod
    def _print_file(filepath):
        if os.path.exists(filepath):
            response = open(filepath).read().decode("utf-8")
            logging.info("{} out:\n{}".format(filepath, response))
        else:
            logging.info("no output for {}".format(filepath))

    def _fill_yql_template(self, yql_tpl):
        target_date, env_type = self._get_task_params()
        yql_tpl = yql_tpl.replace("[[TARGET_DATE]]", target_date)

        week_ago_date = (date.today() - timedelta(days=7)).strftime('%Y-%m-%d')
        yql_tpl = yql_tpl.replace("[[WEEK_AGO_DATE]]", week_ago_date)

        yql = yql_tpl.replace("[[ENV_TYPE]]", env_type)
        return yql

    def _get_task_params(self):
        target_date = self.Parameters.target_date or date.today().strftime('%Y-%m-%d')
        env_type = self.Parameters.env_type or "production"
        return target_date, env_type

    @staticmethod
    def _prepare_venv(venv):
        venv.pip("setuptools")
        venv.pip("urllib3==1.24.3")
        venv.pip("requests")
        environments.PipEnvironment('yql', venv=venv, version='1.2.93').prepare()
        environments.PipEnvironment('yandex-yt-yson-bindings-skynet', venv=venv, use_wheel=True).prepare()
        environments.PipEnvironment('yandex-yt', venv=venv, use_wheel=True).prepare()

    @staticmethod
    def _write_to_file(file_path, content):
        with io.open(file_path, 'w', encoding='utf-8') as fp:
            fp.write(content)

    @staticmethod
    def _get_yql_token():
        owner = "robot-market-ugc-bot"
        name = "YQL_TOKEN"
        return unicode(sdk2.Vault.data(owner, name), 'utf-8')
