import logging
import os
import shutil

from sandbox import sdk2
from sandbox.common.types.client import Tag
from sandbox.common.types.misc import DnsType
from sandbox.projects.common.arcadia import sdk
from sandbox.projects.common.build.parameters import ArcadiaUrl, BASE_BUILD_GROUP_NAME, RESOURCES_GROUP_NAME
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.projects.market.front.helpers.sandbox_helpers import rich_check_call
from sandbox.projects.mobile_apps.utils.android_sdk_env import AndroidSdkEnvironment
from sandbox.projects.sandbox_ci import SANDBOX_CI_ARTIFACT

logger = logging.getLogger(__name__)


class WmsAndroidBuild(sdk2.Task):

    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = Tag.Group.LINUX
        dns = DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):
        checkout_arcadia_from_url = ArcadiaUrl()
        with sdk2.parameters.Group(BASE_BUILD_GROUP_NAME) as git_block:
            android_sdk = sdk2.parameters.String("Android SDK", required=True)
            version_code = sdk2.parameters.String("Version Code")
            gradle_command = sdk2.parameters.String("Gradle command", required=True, default="build")
            work_dir = sdk2.parameters.String("Path to build", required=True)
        with sdk2.parameters.Group(RESOURCES_GROUP_NAME) as resources:
            result_dir = sdk2.parameters.String("Path to result")
            html_page_resources = sdk2.parameters.List(
                'Path to HTML dir',
                sdk2.parameters.String
            )

    class Context(sdk2.Context):
        pass

    def on_execute(self):
        self._prepare()
        with self._get_arcadia() as arcadia_src_dir:
            self.Context.arcadia_src_dir = arcadia_src_dir
            self._do_build()
            self._finish()

    def _prepare(self):
        android_sdk_environment = AndroidSdkEnvironment(self.Parameters.android_sdk)
        android_sdk_environment.prepare(None, 1)

    def _get_arcadia(self):
        return sdk.mount_arc_path(self.Parameters.checkout_arcadia_from_url)

    def _do_build(self):
        cwd = os.path.join(self.Context.arcadia_src_dir, self.Parameters.work_dir)
        rich_check_call(
            ['./gradlew'] + self.Parameters.gradle_command.split(" "),
            task=self, alias="gradle_command", cwd=cwd
        )

    def _create_resource(self):
        logging.info("\nCreating resources")
        for root, dirs, files in os.walk(os.path.join(self.Context.arcadia_src_dir, self.Parameters.work_dir, self.Parameters.result_dir)):
            for file in files:
                if file.endswith(".apk"):
                    shutil.copy(os.path.join(root, file), file)
                    sdk2.ResourceData(SANDBOX_CI_ARTIFACT(
                        self, file, file
                    )).ready()

    def _create_html_resource(self, dir_path, description):
        resource_path = os.path.join(self.Context.arcadia_src_dir, self.Parameters.work_dir, dir_path)
        if os.path.exists(resource_path):
            shutil.copytree(resource_path, dir_path, symlinks=True)
            resource = SANDBOX_CI_ARTIFACT(
                task=self,
                description=description,
                path=dir_path,
            )
            sdk2.ResourceData(resource).ready()

    def _finish(self):
        self._create_resource()
        for path in self.Parameters.html_page_resources:
            self._create_html_resource(path, path)
