# coding=utf-8
import logging
import os

from sandbox import sdk2

import sandbox.common.types.client as ctc
from sandbox.common.types import resource as ctr
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common import binary_task
from sandbox.projects.common.vcs import arc
from sandbox.projects.market.wms.resources import resources as wms


class WmsArcMigrationsGeneral(binary_task.LastBinaryTaskRelease, sdk2.Task):
    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('17'),
        )
        client_tags = ctc.Tag.Group.LINUX

    class Parameters(sdk2.Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group("Arc Settings") as arc_block:
            checkout_path = sdk2.parameters.String("Arcadia checkout path", required=True)
            branch = sdk2.parameters.String("Arc branch", required=False, default=None)

        with sdk2.parameters.Group('Database Settings') as settings_block:
            url = sdk2.parameters.String("Connection url", required=True)
            username = sdk2.parameters.YavSecret("Database username", required=True)
            password = sdk2.parameters.YavSecret("Database password", required=True)

        with sdk2.parameters.Group('Liquibase Settings') as liquibase_block:
            changelog_file = sdk2.parameters.String("Changelog path", required=True)
            liquibase_resource = sdk2.parameters.Resource("Liquibase resource",
                                                          resource_type=wms.WMS_LIQUIBASE,
                                                          required=True,
                                                          default_value=2106885217)
            with sdk2.parameters.String("Liquibase command", multiline=True) as command:
                command.values.update = command.Value(default=True)
                command.values.rollback = None
            tag = sdk2.parameters.String("Rollback tag")
            options = sdk2.parameters.List("Liquibase options",
                                           required=True,
                                           default=["--logLevel=debug"])

    @property
    def binary_executor_query(self):
        return {
            "attrs": {
                "task_type": "WMS_ARC_MIGRATIONS_GENERAL",
                "released": self.Parameters.binary_executor_release_type
            },
            "state": [ctr.State.READY]
        }

    @staticmethod
    def _format_liquibase_command(liquibase_path, migration_file, jdbc_url, username,
                                  password, command, options):
        return '{liquibase_path}/liquibase ' \
               '--changeLogFile={migration_file} ' \
               '--url="{jdbc_url}" --username={username} --password={password} {options} {cmd}' \
            .format(liquibase_path=liquibase_path,
                    migration_file=migration_file,
                    jdbc_url=jdbc_url,
                    username=username,
                    password=password,
                    cmd=command,
                    options=" ".join(options))

    @property
    def resource_path(self):
        return str(sdk2.ResourceData(self.Parameters.liquibase_resource).path)

    @property
    def username(self):
        return self.Parameters.username.data()[self.Parameters.username.default_key]

    @property
    def password(self):
        return self.Parameters.password.data()[self.Parameters.password.default_key]

    def on_execute(self):
        super(WmsArcMigrationsGeneral, self).on_execute()

        branch = self.Parameters.branch
        checkout_path = self.Parameters.checkout_path[1:] \
            if self.Parameters.checkout_path[0] == '/' else self.Parameters.checkout_path

        connect_url = self.Parameters.url
        changelog_file = self.Parameters.changelog_file

        options = self.Parameters.options
        command = self.Parameters.command

        logging.info("Mount arcadia")

        with arc.Arc().mount_path(None, changeset=branch, fetch_all=False) as mount_dir:
            os.chdir(os.path.join(mount_dir, checkout_path))

            logging.info("Change current dir: {dir}".format(dir=os.path.join(mount_dir, checkout_path)))
            liquibase_command = self._format_liquibase_command(self.resource_path, changelog_file, connect_url,
                                                               self.username, self.password, command, options)

            logging.info("Liquibase command: {command}".format(command=liquibase_command))
            logging.info("Run liquibase command")

            with sdk2.helpers.ProcessLog(self, logger='liquibase') as pl:
                sp.check_call([liquibase_command], shell=True, stdout=pl.stdout, stderr=pl.stderr)

            logging.info("Migrations finished")
