# coding=utf-8
import uuid
import logging

from datetime import datetime

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox import sdk2
from sandbox.sdk2 import Task
from sandbox.projects.common import binary_task
from sandbox.common.types import resource as ctr

QUERY = """insert into autotests_launches (*) values ('{uuid}', '{author}', '{now}', '{revision}',
           '{test_name}', '{status}', '{resource}', '{sandbox_task}');
"""


class WmsAutotestsStatCollector(binary_task.LastBinaryTaskRelease, Task):

    class Requirements(Task.Requirements):
        dns = ctm.DnsType.DNS64
        client_tags = ctc.Tag.MULTISLOT

    class Parameters(Task.Parameters):
        ext_params = binary_task.binary_release_parameters(stable=True)

        with sdk2.parameters.Group("Autotests data") as autotests_block:
            author = sdk2.parameters.String("Автор", required=True)
            revision = sdk2.parameters.String("Ревизия", required=True)
            test_name = sdk2.parameters.String("Название тестов", required=True)
            result_status = sdk2.parameters.String("Результат", required=True)
            resource = sdk2.parameters.String("Allure ресурс", required=True)
            sandbox_task_id = sdk2.parameters.String("Sandbox task id", required=True)

        with sdk2.parameters.Group("Clickhouse settings") as clickhouse_block:
            clickhouse_replicas = sdk2.parameters.List("Clickhouse replicas", required=True,
                                                       default=[
                                                           "man-1l1zkqer7q9wf4tn.db.yandex.net",
                                                           "sas-0lk2jl2y3v5n6eje.db.yandex.net",
                                                           "vla-9pqkqgrk7wxsvans.db.yandex.net",
                                                       ])
            clickhouse_user = sdk2.parameters.YavSecret("Clickhouse user", required=True)
            clickhouse_pass = sdk2.parameters.YavSecret("Clickhouse pass", required=True)
            clickhouse_db = sdk2.parameters.String("Clickhouse db name", default="wms", required=True)

    @property
    def binary_executor_query(self):
        return {
            "attrs": {"task_type": "WMS_AUTOTESTS_STAT_COLLECTOR",
                      "released": self.Parameters.binary_executor_release_type},
            "state": [ctr.State.READY]
        }

    def on_execute(self):
        binary_task.LastBinaryTaskRelease.on_execute(self)
        from clickhouse_driver import Client

        replicas = self.Parameters.clickhouse_replicas
        db = self.Parameters.clickhouse_db
        login = self.Parameters.clickhouse_user.data()[self.Parameters.clickhouse_user.default_key]
        password = self.Parameters.clickhouse_pass.data()[self.Parameters.clickhouse_pass.default_key]

        author = self.Parameters.author
        now = datetime.now().strftime("%Y-%m-%d %H:%M:%S")
        revision = self.Parameters.revision
        test_name = self.Parameters.test_name
        status = self.Parameters.result_status
        resource = self.Parameters.resource
        sandbox_task_id = self.Parameters.sandbox_task_id

        query = QUERY.format(uuid=uuid.uuid4(), author=author, now=now, revision=revision, test_name=test_name,
                             status=status, resource=resource, sandbox_task=sandbox_task_id)

        for host in replicas:
            try:
                client = Client.from_url("clickhouses://{login}:{password}@{host}:9440/{db}"
                                         .format(login=login, password=password, host=host, db=db))

                logging.info("Execute query: {query}".format(query=query))
                client.execute(query)
                logging.info("Autotests statistics saved")
                return
            except Exception as e:
                logging.error('Host %s failed: %s', host, e)

        raise RuntimeError('All host failed')
