import logging
import os
import re

from sandbox import sdk2

import sandbox.common.types.client as ctc
import sandbox.common.types.misc as ctm
from sandbox.projects.market.wms.helpers import build_utils as bu
from sandbox.sdk2.helpers import subprocess as sp
from sandbox.projects.common.environments import SandboxJavaJdkEnvironment
from sandbox.sdk2.vcs.git import Git
from sandbox.projects.resource_types import OTHER_RESOURCE as OtherResource
from sandbox.projects import resource_types as rt

logger = logging.getLogger(__name__)


class WmsBuild(sdk2.Task):
    work_dir_name = 'wms'

    class Requirements(sdk2.Task.Requirements):
        environments = (
            SandboxJavaJdkEnvironment('11.0.2'),
        )
        client_tags = ctc.Tag.Group.LINUX
        dns = ctm.DnsType.DNS64

    class Parameters(sdk2.Task.Parameters):

        with sdk2.parameters.Group('Git Settings') as git_block:
            repository = sdk2.parameters.String("SSH Git repository", required=True)
            git_ssh = sdk2.parameters.YavSecret("SSH key", default="sec-01dgd1q7tdynkch7sdcf1taxyt", required=True)
            branch = sdk2.parameters.String("Branch")
            commit = sdk2.parameters.String("Commit")
            filter_branches = sdk2.parameters.Bool("Limit to refs/heads, refs/tags", default=True)

        with sdk2.parameters.Group('Build Settings') as git_block:
            build_id = sdk2.parameters.String("Build ID", default="0")
            release_type = sdk2.parameters.String("Release type")
            tasks = sdk2.parameters.List("Gradle tasks", required=True, default=['clean', 'releaseTar'])
            parallel_execution = sdk2.parameters.Bool("Allow Gradle parallel execution", default=False)
            excluded_tasks = sdk2.parameters.List("Excluded Gradle tasks", required=False)

    class Context(sdk2.Task.Context):
        execution_number = 0

    def on_execute(self):
        self.Context.execution_number += 1

        logger.info("Clone git repository {repository}".format(
            repository=self.Parameters.repository
        ))
        ssh_key = self.Parameters.git_ssh.data()['ssh-private']
        with sdk2.ssh.Key(self, private_part=ssh_key):
            git = Git(self.Parameters.repository, filter_branches=self.Parameters.filter_branches)
            git.clone(target_dir=self.work_dir_name, branch=self.Parameters.branch, commit=self.Parameters.commit)
        try:
            self.do_build()
            self.create_resource()
        finally:
            self._collect_and_publish_reports()

    def do_build(self):
        with sdk2.helpers.ProcessLog(self, logger='build') as pl:
            cmd = ['./gradlew',
                    '-Pbuild.number={}'.format(self.Parameters.build_id),
                    '-Pinfor.release.type={}'.format(self.Parameters.release_type),
                    '--stacktrace']

            if self.Parameters.parallel_execution:
                cmd += ['--parallel']

            for task in self.Parameters.tasks:
                cmd += [task]

            for excluded_task in self.Parameters.excluded_tasks:
                cmd += ['-x', excluded_task]

            sp.check_call(cmd, cwd=self.work_dir_name, stdout=pl.stdout, stderr=pl.stderr)
            self.set_info('Modules have been built')

    def create_resource(self):
        path = self._find_release_tar()
        logger.info('Creating resource from path {}'.format(path))
        self._create_resource(path)

    def _create_resource(self, path):
        resource = OtherResource(self, 'Release TAR', path, resource_name='infor-release-tar', resource_version=self.Parameters.build_id)
        resource.ttl = 14
        data = sdk2.ResourceData(resource)
        data.ready()
        logger.info('Resource {} created'.format(resource.id))

    def _find_release_tar(self):
        package_pattern = re.compile(r'infor-.*-b.+-[^-]+\.tar')
        dist_dir = os.path.join(self.work_dir_name, 'build/distributions')
        logger.info('Looking for release TAR file in {}'.format(dist_dir))
        for root, _, files in os.walk(dist_dir):
            for f_file in files:
                if package_pattern.match(f_file):
                    logger.info('Found release TAR: {}'.format(f_file))
                    return os.path.join(root, f_file)

    def _collect_and_publish_reports(self):
        reports_path = "build_reports{}".format(self.Context.execution_number)
        bu.copy_reports(self.work_dir_name, reports_path)

        if (os.path.exists(reports_path)):
            resource = rt.BUILD_OUTPUT(self, 'Build reports', reports_path, resource_name='build-reports')
            resource.ttl = 14
            data = sdk2.ResourceData(resource)
            data.ready()
            logger.info('Resource {} created'.format(resource.id))
