import logging
import os
import errno
import tarfile

import sandbox.projects.common.file_utils as fu
import sandbox.projects.common.utils as utils
from sandbox import sdk2
from sandbox.projects.common import apihelpers
from sandbox.projects import resource_types
from sandbox.sdk2.helpers import subprocess as sp

logger = logging.getLogger(__name__)


class WmsDatasourcesMultitesting(sdk2.Resource):
    releasable = True
    backup_task = True


def ensure_directory_exists(directory):
    try:
        os.makedirs(directory)
    except OSError as err:
        if err.errno != errno.EEXIST:
            raise


def write_confd_config_file(
        confd_config_file_path,
        etcd_keys,
        etcd_prefix,
        file_mode,
        output_file_path,
        template_name
):
    contents = """
            [template]
            keys = {}
            prefix = "{}"
            mode = "{}"
            src = "{}"
            dest = "{}"
            """.format(etcd_keys, etcd_prefix, file_mode, template_name, output_file_path)
    logging.info("Writing confd template resource to %s: %s", confd_config_file_path, contents)
    fu.write_file(confd_config_file_path, contents)


class WmsBuildDatasources(sdk2.Task):
    """A task that packs Market Datasources into tarball"""

    class Parameters(sdk2.Task.Parameters):

        vault_key = sdk2.parameters.String(
            "Vault key, containing etcd password",
            default="market_etcd_datasources_read_all",
            required=True
        )
        etcd_username = sdk2.parameters.String("etcd username", default="datasources_read_all", required=True)
        multitesting_etcd_prefix = sdk2.parameters.String(
            'Multitesting etcd prefix',
            required=False,
            description='Example: /multitesting/front-desktop--my_awesome_mt/.',
        )
        multitesting_parent_environment_name = sdk2.parameters.String(
            'Multitesting environment',
            required=False,
            default="testing",
            description='testing/prestable/production',
        )
        ignore_not_existing_files = sdk2.parameters.Bool(
            'Ignore not existing files',
            required=True,
            default=False
        )

    def on_execute(self):
        logger.info("""Parameters picked:
        Etcd username: {}
        Multitesting etcd prefix: {}
        Multitesting parent environment name: {}
        Ignore not existing files: {}
        """.format(self.Parameters.etcd_username,
                   self.Parameters.multitesting_etcd_prefix,
                   self.Parameters.multitesting_parent_environment_name,
                   self.Parameters.ignore_not_existing_files
                   ))

        os.makedirs("confd/templates")
        template_content = """
        {{- $ir_datasources_prefix := print "/wms/" (getenv "YANDEX_ENVIRONMENT") }}
        {{ range gets (print $ir_datasources_prefix "/*") }}{{ base .Key }}={{ .Value }}
        {{ end }}"""

        logging.info('Writing config template: {}'.format(template_content))
        with open('confd/templates/datasources.properties.tmpl', 'w+') as f:
            f.write(template_content)

        self.confd_directory = os.path.join(os.getcwd(), 'confd')
        self._write_confd_config()

        self._create_datasources_resource(
            resource_type=WmsDatasourcesMultitesting,
            environment_name=self.Parameters.multitesting_parent_environment_name,
            multitesting_etcd_prefix=self.Parameters.multitesting_etcd_prefix,
        )

    def _write_confd_config(self):
        confd_toml_content = """
                          backend = "{}"
                          basic_auth = {}
                          nodes = ["{}"]
                          client_cakeys = "{}"
                          """.format('etcdv3',
                                     'true',
                                     'etcd.vs.market.yandex.net:3379',
                                     '/usr/share/yandex-internal-root-ca/YandexInternalRootCA.crt')

        logging.info('Writing cond config: {}'.format(confd_toml_content))
        with open('confd/confd.toml', 'w+') as f:
            f.write(confd_toml_content)

    def _create_datasources_resource(self, resource_type, environment_name, multitesting_etcd_prefix='/', dir_if_deploy=None):

        common_etcd_prefix = os.path.join(multitesting_etcd_prefix, 'datasources')

        if dir_if_deploy:
            datasources_path = dir_if_deploy
            archive_name = 'datasources-{}-deploy.tar.gz'.format(environment_name)
        else:
            datasources_path = environment_name
            archive_name = 'datasources-{}.tar.gz'.format(environment_name)

        logging.info(
            'Creating resource %s for environment %s from etcd prefix %s',
            resource_type.__name__,
            environment_name,
            common_etcd_prefix
        )

        input_file_names = [
            self._create_datasources_file(
                environment_name=environment_name,
                etcd_prefix=common_etcd_prefix,
                etcd_keys=["/wms/{}".format(environment_name)],
                template_name='datasources.properties.tmpl',
                output_file_path=os.path.join(
                    datasources_path,
                    'properties.d',
                    'override.properties'
                ),
            ),
        ]

        self._create_datasources_resource_from_files(
            resource_type=resource_type,
            tarball_file_name=archive_name,
            input_file_names=input_file_names
        )

    def _create_datasources_file(self, environment_name, etcd_prefix, template_name, output_file_path, etcd_keys=None,
                                 file_mode='0644'):
        etcd_keys = etcd_keys or ['/']

        logging.info('Creating file %s from template %s and etcd prefix %s',
                     output_file_path, template_name, etcd_prefix)

        self._prepare_confd_directory(
            etcd_prefix=etcd_prefix,
            template_name=template_name,
            output_file_path=output_file_path,
            etcd_keys=etcd_keys,
            file_mode=file_mode,
        )

        ensure_directory_exists(os.path.dirname(output_file_path))
        self._run_confd(environment_name=environment_name)

        return output_file_path

    def _run_confd(self, environment_name):
        run_env = os.environ.copy()
        run_env["ETCD_PASSWORD"] = sdk2.Vault.data(self.Parameters.vault_key)
        run_env["YANDEX_ENVIRONMENT"] = environment_name
        logging.info('YANDEX_ENVIRONMENT=%s', environment_name)
        with sdk2.helpers.ProcessLog(self, logger=logging.getLogger("run_confd")) as pl:
            sp.Popen(
                "{} -log-level info -onetime -config-file confd/confd.toml -confdir confd/"
                " -sync-only -username {} -password $ETCD_PASSWORD".format(
                    self._get_confd_executable_path(), self.Parameters.etcd_username
                ),
                shell=True,
                stdout=pl.stdout,
                stderr=pl.stdout,
                env=run_env
            ).communicate()

    def _get_confd_executable_path(self):
        resource_id = apihelpers.get_last_resource_with_attribute(resource_types.CONFD_BIN).id
        logging.info('CONFD_BIN resource id: %s', resource_id)
        resource_path = utils.sync_resource(resource_id)
        logging.info('CONFD_BIN resource path: %s', resource_path)

        return resource_path

    def _prepare_confd_directory(self, etcd_prefix, template_name, output_file_path, etcd_keys, file_mode):
        confd_directory = self.confd_directory

        template_config_directory = os.path.join(confd_directory, 'conf.d')
        template_config_path = os.path.join(template_config_directory, 'config.toml')

        ensure_directory_exists(template_config_directory)
        write_confd_config_file(
            confd_config_file_path=template_config_path,
            etcd_keys=etcd_keys,
            etcd_prefix=etcd_prefix,
            file_mode=file_mode,
            output_file_path=os.path.join(os.getcwd(), output_file_path),
            template_name=template_name
        )

        return confd_directory

    def _create_datasources_resource_from_files(self, resource_type, tarball_file_name, input_file_names):
        data = sdk2.ResourceData(resource_type(self, 'Market Datasources', tarball_file_name))

        logging.info('Creating tarball %s', tarball_file_name)
        with tarfile.open(name=tarball_file_name, mode='w:gz') as tar:
            for input_file_name in input_file_names:
                logging.info('Adding %s to tarball', input_file_name)
                if (not self.Parameters.ignore_not_existing_files or os.path.exists(input_file_name)):
                    tar.add(input_file_name)
                else:
                    logging.warn('%s does not exists', input_file_name)

        logging.info('Marking resource %s as ready', resource_type.type)
        data.ready()
